/* Copyright © 2024 Com2uS Platform Corp. All Rights Reserved. */
#include "PlayerTestView.h"
#include "HIVESDKV4Tester.h"
#include "HIVESDKV4TesterGameMode.h"
#include <sstream>
UPlayerTestView::UPlayerTestView(const FObjectInitializer& ObjectInitializer)
: Super(ObjectInitializer)
{
    bAutoSignIn = false;

    InitAutoLoginEnabled();
}

void UPlayerTestView::InitAutoLoginEnabled() {
#if PLATFORM_ANDROID || PLATFORM_IOS
    bAutoLoginEnabled = true;
#else
    bAutoLoginEnabled = false;
#endif
}

void PrintFHivePlayerInfo(const FHivePlayerInfo& PlayerInfo, ELogType targetLogType)
{
    FString Message = FString::Printf(TEXT("PlayerId: %lld\nPlayerName: %s\nPlayerImageUrl: %s\nPlayerToken: %s\nAccessToken: %s\nDid: %s\n"),
                                      PlayerInfo.PlayerId,
                                      *(PlayerInfo.PlayerName),
                                      *(PlayerInfo.PlayerImageUrl),
                                      *(PlayerInfo.PlayerToken),
                                      *(PlayerInfo.AccessToken),
                                      *(PlayerInfo.Did));
    
    Message += TEXT("[ProviderInfoData]\n");
    for (const auto& PairProviderInfo : PlayerInfo.ProviderInfoData)
    {
        FHiveProviderInfo ProviderInfo = PairProviderInfo.Value;
        Message += FString::Printf(TEXT("ProviderInfo.ProviderType: %s\n"), *(GetNameFromEHiveProviderType(ProviderInfo.ProviderType)));
        Message += FString::Printf(TEXT("ProviderInfo.ProviderName: %s\n"), *(ProviderInfo.ProviderName));
        Message += FString::Printf(TEXT("ProviderInfo.ProviderUserId: %s\n"), *(ProviderInfo.ProviderUserId));
        Message += FString::Printf(TEXT("ProviderInfo.ProviderEmail: %s\n\n"), *(ProviderInfo.ProviderEmail));
    }
    
    Message += TEXT("[CustomProviderInfoData]\n");
    for (const auto& PairCustomProviderInfo : PlayerInfo.CustomProviderInfoData)
    {
        FHiveProviderInfo CustomProviderInfo = PairCustomProviderInfo.Value;
        Message += FString::Printf(TEXT("ProviderInfo.ProviderType: %s\n"), *(GetNameFromEHiveProviderType(CustomProviderInfo.ProviderType)));
        Message += FString::Printf(TEXT("ProviderInfo.ProviderName: %s\n"), *(CustomProviderInfo.ProviderName));
        Message += FString::Printf(TEXT("ProviderInfo.ProviderUserId: %s\n"), *(CustomProviderInfo.ProviderUserId));
        Message += FString::Printf(TEXT("ProviderInfo.ProviderEmail: %s\n\n"), *(CustomProviderInfo.ProviderEmail));
    }
    
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(targetLogType ,TEXT("%s"), *Message);
}


void ProcessNeedExit(FHiveResultAPI Result, ELogType targetLogType)
{
    if (Result.NeedExit())
    {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(targetLogType ,TEXT("ResultAPI.needExit() = true"));
        
        if(AHIVESDKV4TesterGameMode::getGameModeInstance()->OnExitPopupSampleForAndroid.IsBound())
        {
            AHIVESDKV4TesterGameMode::getGameModeInstance()->OnExitPopupSampleForAndroid.Broadcast();
        }
    }
}


void UPlayerTestView::PlayerSetup()
{
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("=== Setup() ==="));
    
    FHiveAuthV4::Setup(FHiveAuthV4OnSetupDelegate::CreateLambda([this](const FHiveResultAPI& Result,
                                                               bool IsAutoSignIn,
                                                               const FString& Did,
                                                               const TArray<EHiveProviderType>& ProviderTypeArray) {
        
        this->bAutoSignIn = IsAutoSignIn;
        
        FString ProviderTypeArrayString;
        for (auto& ProviderType : ProviderTypeArray)
        {
            ProviderTypeArrayString.Append(GetNameFromEHiveProviderType(ProviderType));
            ProviderTypeArrayString.Append(TEXT("\n"));
        }
        
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4,
                                                                         TEXT("Result : %s\nisAutoSignIn : %s\nDID : %s\n[EHiveProviderType]\n%s"),
                                                                         *(Result.ToString()),
                                                                         IsAutoSignIn ? TEXT("TRUE") : TEXT("FALSE"),
                                                                         *Did,
                                                                         *ProviderTypeArrayString);
        if(Result.IsSuccess())
        {
            AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("== SetEngagementDelegate() ==="), *(Result.ToString()));
            FHivePromotion::SetEngagementDelegate(FHivePromotionOnEngagementDelegate::CreateLambda([this](const FHiveResultAPI& Result,
                                                                                                      const EHiveEngagementEventType& EngagementEventType,
                                                                                                      const EHiveEngagementEventState& EngagementEventState,
                                                                                                      TSharedPtr<FJsonObject> Param) {
                
                // FJsonObject를 문자열로 직렬화
                FString ParamFString;
                TSharedRef<TJsonWriter<>> JsonWriter = TJsonWriterFactory<>::Create(&ParamFString);
                FJsonSerializer::Serialize(Param.ToSharedRef(), JsonWriter);
                
                AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("%s"), *(Result.ToString()));
                AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4,
                                                                                 TEXT("EngagementEventType = %s\nEngagementEventState = %s\nParam = %s"),
                                                                                 *(GetNameFromEHiveEngagementEventType(EngagementEventType)),
                                                                                 *(GetNameFromEHiveEngagementEventState(EngagementEventState)),
                                                                                 *ParamFString);
                
                switch (EngagementEventType)
                {
                    case EHiveEngagementEventType::EVENT_TYPE: {
                        if (EngagementEventState == EHiveEngagementEventState::BEGIN) {
                            // Engagement의 모든 동작 수행됨
                        } else if (EngagementEventState == EHiveEngagementEventState::FINISH) {
                            // Engagement의 모든 동작 수행 완료됨
                        }
                    }
                        break;
                        
                    case EHiveEngagementEventType::AUTH_LOGIN_VIEW: {
                        if (EngagementEventState == EHiveEngagementEventState::EVENT_START) {
                            // Hive 멤버십 로그인 페이지 노출 요청 시작
                        } else if (EngagementEventState == EHiveEngagementEventState::EVENT_END) {
                            // Hive 멤버십 로그인 수행 완료 (로그인 완료/창닫음 수행 이후 전달)
                        }
                    }
                        break;
                        
                    case EHiveEngagementEventType::SOCIAL_INQUIRY_VIEW: {
                        if (EngagementEventState == EHiveEngagementEventState::EVENT_START) {
                            // 1:1 문의 페이지 노출 요청 시작
                        } else if (EngagementEventState == EHiveEngagementEventState::EVENT_END) {
                            // 1:1 문의 페이지 노출 완료 (1:1문의 창 닫은 이후 전달)
                        }
                    }
                        break;
                        
                    case EHiveEngagementEventType::PROMOTION_VIEW: {
                        if (EngagementEventState == EHiveEngagementEventState::EVENT_START) {
                            // Promotion View 노출 요청 시작
                        } else if (EngagementEventState == EHiveEngagementEventState::EVENT_END) {
                            // Promotion View 노출 완료 (Promotion 창 닫은 이후 전달)
                        }
                    }
                        break;
                        
                    case EHiveEngagementEventType::COUPON: {
                        if (EngagementEventState == EHiveEngagementEventState::EVENT_START) {
                            // 쿠폰 소모 요청 시작
                        } else if (EngagementEventState == EHiveEngagementEventState::EVENT_END) {
                            // 쿠폰 소모 요청 완료 및 응답 전달
                        }
                    }
                        break;
                        
                    case EHiveEngagementEventType::OFFERWALL_VIEW: {
                        if (EngagementEventState == EHiveEngagementEventState::EVENT_START) {
                            // Offerwall 페이지 노출 요청 시작
                        } else if (EngagementEventState == EHiveEngagementEventState::EVENT_END) {
                            // Offerwall 페이지 노출 완료 (Offerwall 창 닫은 이후 전달)
                        }
                    }
                        break;
                        
                    case EHiveEngagementEventType::EVENT: {
                        if (EngagementEventState == EHiveEngagementEventState::EVENT_START) {
                            // host가 game인 경우 해당 데이터 전달
                        } else if (EngagementEventState == EHiveEngagementEventState::EVENT_END) {
                            // host가 game인 경우 해당 데이터 전달
                        }
                    }
                        break;
                        
                    case EHiveEngagementEventType::IAP_UPDATED: {
                        if (EngagementEventState == EHiveEngagementEventState::EVENT_START) {
                            // 구매내역이 변경되었으니 받아가라는 ( Restore() 를 호출 ) 신호
                        } else if (EngagementEventState == EHiveEngagementEventState::EVENT_END) {
                            // 구매내역이 변경되었으니 받아가라는 ( Restore() 를 호출 ) 신호
                        }
                    }
                        break;
                        
                    case EHiveEngagementEventType::IAP_PROMOTE: {
                        if (EngagementEventState == EHiveEngagementEventState::EVENT_START) {
                            // IAP Promote 데이터 존재 여부 전달
                        } else if (EngagementEventState == EHiveEngagementEventState::EVENT_END) {
                            // IAP Promote 데이터 존재 여부 전달
                        }
                    }
                        break;
                        
                    case EHiveEngagementEventType::IAP_PURCHASE: {
                        if (EngagementEventState == EHiveEngagementEventState::EVENT_START) {
                            // IAPV4 Purhcase 구매 시작
                        } else if (EngagementEventState == EHiveEngagementEventState::EVENT_END) {
                            // IAPV4 Purhcase 구매 완료 (+영수증데이터)
                        }
                    }
                        break;
                        
                    case EHiveEngagementEventType::SOCIAL_MYINQUIRY_VIEW: {
                        if (EngagementEventState == EHiveEngagementEventState::EVENT_START) {
                            // 내 문의 페이지 노출 시작
                        } else if (EngagementEventState == EHiveEngagementEventState::EVENT_END) {
                            // 내 문의 페이지 노출 완료
                        }
                    }
                        break;
                        
                    case EHiveEngagementEventType::SOCIAL_PROFILE_VIEW: {
                        if (EngagementEventState == EHiveEngagementEventState::EVENT_START) {
                            // 프로필 페이지 노출 시작
                        } else if (EngagementEventState == EHiveEngagementEventState::EVENT_END) {
                            // 프로필 페이지 노출 완료
                        }
                    }
                        break;
                        
                    default:
                        break;
                }
                
            }));
        }
        
        ProcessNeedExit(Result, ELogType::AuthV4);
#if !PLATFORM_ANDROID && !PLATFORM_IOS
        bAutoLoginEnabled = IsAutoSignIn;
        if (AHIVESDKV4TesterGameMode::getGameModeInstance()->OnUpdateSetAutoLoginEnabled.IsBound())
        {
            AHIVESDKV4TesterGameMode::getGameModeInstance()->OnUpdateSetAutoLoginEnabled.Broadcast();
        }
#endif
        }));
}
bool UPlayerTestView::PlayerGetSignInType()
{
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("SignIn Type (Auto SignIn) : %s\n"), bAutoSignIn?TEXT("TRUE"):TEXT("FALSE"));
    return bAutoSignIn;
}

void UPlayerTestView::PlayerSignIn(FString providerType)
{
    EHiveProviderType ProviderType = GetEHiveProviderTypeFromName(providerType);
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("=== SignIn(%s) ==="), *providerType);
    FHiveAuthV4::SignIn(ProviderType,
                        FHiveAuthV4OnSignInDelegate::CreateLambda([this](const FHiveResultAPI& Result, const FHivePlayerInfo& PlayerInfo) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("SignIn result : %s"), *(Result.ToString()));
        if (Result.IsSuccess())
        {
            this->mCurrentPlayer = PlayerInfo;
            PrintFHivePlayerInfo(PlayerInfo, ELogType::AuthV4);
            RequestPostBoxLogin(); // Request PostBox Login
        }
        ProcessNeedExit(Result, ELogType::AuthV4);
    }));
}

void UPlayerTestView::SignInWithAuthKey(FString customProviderName, FString customProviderUserId) {
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("=== SignInWithAuthKey(%s, %s) ==="), *customProviderName, *customProviderUserId);
    
    FString authType = TEXT("SIGNINIDP");
    
    TestAppGameNetwork::FGameListener listener;
    
    if(listener.IsBound())
    {
        listener.Unbind();
    }
    
    listener.BindUObject(this, &UPlayerTestView::postCustomAuthKey);
    
    TestAppGameNetwork::customAuth(customProviderName, customProviderUserId, authType, listener);
}

void UPlayerTestView::postCustomAuthKey(bool result, ITestAppGameNetworkResponse response) {
    if(result)
    {
        UE_LOG(LogTemp, VeryVerbose, TEXT("success postCustomAuthKey()"));
       
        FString AuthKey = response.authKey;
        
        FHiveAuthV4::SignInWithAuthKey(AuthKey, FHiveAuthV4OnSignInDelegate::CreateLambda([this](const FHiveResultAPI& Result, const FHivePlayerInfo& PlayerInfo) {
            
            AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("signInWithAuthKey result : %s"), *(Result.ToString()));
            
            if(Result.IsSuccess())
            {
                this->mCurrentPlayer = PlayerInfo;
                PrintFHivePlayerInfo(PlayerInfo, ELogType::AuthV4);
                RequestPostBoxLogin(); // Request PostBox Login
            }
        }));
    }
    else
    {
        UE_LOG(LogTemp, VeryVerbose, TEXT("fail postCustomAuthKey()"));
    }
}

//-
// 
void UPlayerTestView::postCustomAuthKeyConnect(bool result, ITestAppGameNetworkResponse response) {
    if (result) {
        FString AuthKey = response.authKey;
        FHiveAuthV4::ConnectWithAuthKey(AuthKey, FHiveAuthV4OnConnectDelegate::CreateLambda([this](const FHiveResultAPI& Result, const FHivePlayerInfo& ConflictPlayerInfo) {
            
            AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("ConnectWithAuthKey\nFHiveResultAPI : %s\nplayer ID : %lld\n"), *(Result.ToString()), ConflictPlayerInfo.PlayerId);
            
            if(Result.ErrorCode == FHiveResultAPI::EErrorCode::CONFLICT_PLAYER)
            {
                mConflictPlayer = ConflictPlayerInfo;
                PrintFHivePlayerInfo(ConflictPlayerInfo, ELogType::AuthV4);
            }
            
            ProcessNeedExit(Result, ELogType::AuthV4);
        }));
            
    }
    else {
        UE_LOG(LogTemp, VeryVerbose, TEXT("fail postCustomAuthKeyConnect()"));
    }
}
//-
void UPlayerTestView::PlayerSignOut()
{
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("=== SignOut() ==="));
    FHiveAuthV4::SignOut(FHiveAuthV4OnSignOutDelegate::CreateLambda([this](const FHiveResultAPI& Result) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("Result : %s\n"), *(Result.ToString()));

        InitAutoLoginEnabled();

        if (AHIVESDKV4TesterGameMode::getGameModeInstance()->OnUpdateSetAutoLoginEnabled.IsBound())
        {
            AHIVESDKV4TesterGameMode::getGameModeInstance()->OnUpdateSetAutoLoginEnabled.Broadcast();
        }
    }));
}
    
void UPlayerTestView::PlayerDelete()
{
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("=== PlayerDelete() ==="));
    FHiveAuthV4::PlayerDelete(FHiveAuthV4OnSignOutDelegate::CreateLambda([this](const FHiveResultAPI& Result) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("Result : %s\n"), *(Result.ToString()));
    }));
}

void UPlayerTestView::PlayerConnect(FString providerType)
{
    EHiveProviderType ProviderType = GetEHiveProviderTypeFromName(providerType);
    FHiveAuthV4::Connect(ProviderType,
                     FHiveAuthV4OnConnectDelegate::CreateLambda([this](const FHiveResultAPI& Result, const FHivePlayerInfo& ConflictPlayerInfo) {
        
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("Connect FHiveResultAPI : %s\nplayer ID : %lld\n"), *(Result.ToString()), ConflictPlayerInfo.PlayerId);
        
        if(Result.ErrorCode == FHiveResultAPI::EErrorCode::CONFLICT_PLAYER)
        {
            this->mConflictPlayer = ConflictPlayerInfo;
            PrintFHivePlayerInfo(ConflictPlayerInfo, ELogType::AuthV4);
        }
        
        ProcessNeedExit(Result, ELogType::AuthV4);
    }));
}

void UPlayerTestView::PlayerDisconnect(FString providerType)
{
    EHiveProviderType ProviderType = GetEHiveProviderTypeFromName(providerType);
    FHiveAuthV4::Disconnect(ProviderType, FHiveAuthV4OnDisconnectDelegate::CreateLambda([this](const FHiveResultAPI& Result) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("Disconnect FHiveResultAPI : %s\n"), *(Result.ToString()));
    }));
}

void UPlayerTestView::PlayerConnectWithAuthKey(FString customProviderName, FString customProviderUserId) {

    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("Call AuthV4.connectWithAuthKey()\n"));

    FString authType = TEXT("CONNECT");

    TestAppGameNetwork::FGameListener listener;

    if (listener.IsBound()) {
        listener.Unbind();
    }

    listener.BindUObject(this, &UPlayerTestView::postCustomAuthKeyConnect);

    TestAppGameNetwork::customAuth(customProviderName, customProviderUserId, authType, listener);

}

void UPlayerTestView::PlayerDisconnectWithName(FString customProviderName) {
    FHiveAuthV4::DisconnectWithName(customProviderName, FHiveAuthV4OnDisconnectDelegate::CreateLambda([this](const FHiveResultAPI& Result) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("DisconnectWithName\nFHiveResultAPI : %s\n"), *(Result.ToString()));
    }));
}

void UPlayerTestView::PlayerRefreshAccessToken()
{
    FHiveAuthV4::RefreshAccessToken(FHiveAuthV4OnRefreshAccessTokenDelegate::CreateLambda([this](const FHiveResultAPI& Result, const FString& AccessToken) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("RefreshAccessToken\nFHiveResultAPI : %s\nAccessToken : %s"), *(Result.ToString()), *AccessToken);
    }));
}

void UPlayerTestView::PlayerSelectConflict(FString selectedPlayerId)
{
    int64 SelectedPlayerId;
    LexFromString(SelectedPlayerId, *selectedPlayerId);
    FHiveAuthV4::SelectConflict(SelectedPlayerId,
                                FHiveAuthV4OnSignInDelegate::CreateLambda([this](const FHiveResultAPI& Result, const FHivePlayerInfo& PlayerInfo) {
        
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("SelectConflict\nFHiveResultAPI : %s\n"), *(Result.ToString()));
        
        if (Result.IsSuccess())
        {
            this->mCurrentPlayer = PlayerInfo;
            PrintFHivePlayerInfo(PlayerInfo, ELogType::AuthV4);
            RequestPostBoxLogin(); // Request PostBox Login
        }
    }));
}

void UPlayerTestView::PlayerShowConflictSelection()
{
    /*
     * connect 등 충돌 상황에서 발생되는 PlayerInfo 객체에서는 PlayerId외 별도 정보 없음.
     * 추가 정보 설정을 위해서는 해당 Player Id의 프로필 정보를 조회하여 설정하도록 한다.
     */
    TArray<int64> PlayerIdArray;
    PlayerIdArray.Add(this->mConflictPlayer.PlayerId);
    FHiveAuthV4::GetProfile(PlayerIdArray, FHiveAuthV4OnGetProfileDelegate::CreateLambda([this](const FHiveResultAPI& Result, const TArray<FHiveProfileInfo> ProfileInfoArray) {
        
        FHiveConflictViewData ConflictViewData(this->mCurrentPlayer.PlayerId, this->mConflictPlayer.PlayerId);
        
        if (Result.IsSuccess() && ProfileInfoArray.Num() > 0)
        {
            ConflictViewData.SetData(TEXT("NAME"), this->mCurrentPlayer.PlayerName, ProfileInfoArray[0].PlayerName);
        }
        ConflictViewData.SetData(TEXT("Gold"), TEXT("123456789"), TEXT("456789123"));
        
        FHiveAuthV4::ShowConflictSelection(ConflictViewData,
                                    FHiveAuthV4OnSignInDelegate::CreateLambda([this](const FHiveResultAPI& Result, const FHivePlayerInfo& PlayerInfo) {
            
            AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("ShowConflictSelection\nFHiveResultAPI : %s\n"), *(Result.ToString()));
            
            if (Result.IsSuccess())
            {
                this->mCurrentPlayer = PlayerInfo;
                PrintFHivePlayerInfo(PlayerInfo, ELogType::AuthV4);
                RequestPostBoxLogin(); // Request PostBox Login
            }
        }));
        
    }));
}

void UPlayerTestView::PlayerCheckProvider(FString providerType)
{
    EHiveProviderType ProviderType = GetEHiveProviderTypeFromName(providerType);
    
    FHiveAuthV4::CheckProvider(ProviderType, FHiveAuthV4OnCheckProviderDelegate::CreateLambda([this](const FHiveResultAPI& Result, const FHiveProviderInfo& ProviderInfo) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("CheckProvider\nFHiveResultAPI : %s\n"), *(Result.ToString()));
        
        if (Result.IsSuccess())
        {
            FString ProviderTypeName = GetNameFromEHiveProviderType(ProviderInfo.ProviderType);
            AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("ProviderType: %s\nProviderUserId: %s"), *ProviderTypeName, *(ProviderInfo.ProviderUserId));
        }
    }));
}

FString stringFromMaintenanceAction(AuthV4MaintenanceActionType actionType)
{
    switch(actionType){
        case AuthV4MaintenanceActionType::DONE:       return TEXT("DONE");
        case AuthV4MaintenanceActionType::EXIT:        return TEXT("EXIT");
        case AuthV4MaintenanceActionType::OPEN_URL:    return TEXT("OPEN_URL");
        default:
            return "";
    }
}
void UPlayerTestView::PlayerCheckMaintenance(bool isShow)
{
    FHiveAuthV4::CheckMaintenance(isShow,
                                  FHiveAuthV4OnMaintenanceInfoDelegate::CreateLambda([this](const FHiveResultAPI& Result, const TArray<FHiveAuthV4MaintenanceInfo>& AuthV4MaintenanceInfoArray) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("CheckMaintenance FHiveResultAPI:\n%s\n"), *(Result.ToString()));
        
        if (Result.IsSuccess())
        {
            for (const auto& Info: AuthV4MaintenanceInfoArray)
            {
                FString Message = TEXT("FHiveAuthV4MaintenanceInfo\n");
                Message += FString::Printf(TEXT("Title = %s\n"), *(Info.Title));
                Message += FString::Printf(TEXT("Message = %s\n"), *(Info.Message));
                Message += FString::Printf(TEXT("Button = %s\n"), *(Info.Button));
                Message += FString::Printf(TEXT("Action = %s\n"), *(GetNameFromEHiveAuthV4MaintenanceActionType(Info.Action)));
                Message += FString::Printf(TEXT("Url = %s\n"), *(Info.Url));
                Message += FString::Printf(TEXT("StartDate = %s\n"), *(Info.StartDate));
                Message += FString::Printf(TEXT("EndDate = %s\n"), *(Info.EndDate));
                Message += FString::Printf(TEXT("RemainingTime = %s\n"), *(FString::FromInt(Info.RemainingTime)));
                Message += FString::Printf(TEXT("CustomerButton = %s\n"), *(Info.CustomerButton));
                Message += FString::Printf(TEXT("CustomerLink = %s\n"), *(Info.CustomerLink));
                
                for (const auto& Button : Info.ExButtons)
                {
                    Message += TEXT("= ExtraButton =");
                    Message += FString::Printf(TEXT("Button = %s\n"), *(Button.Button));
                    Message += FString::Printf(TEXT("Action = %s\n"), *(GetNameFromEHiveAuthV4MaintenanceActionType(Button.Action)));
                    Message += FString::Printf(TEXT("Url = %s\n"), *(Button.Url));
                }
                
                Message += TEXT("\n");
                
                AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 , TEXT("%s"), *Message);
            }
        }
        ProcessNeedExit(Result, ELogType::AuthV4);
    }));
}

void UPlayerTestView::PlayerCheckBlacklist(bool isShow)
{
    FHiveAuthV4::CheckBlacklist(isShow,
                                  FHiveAuthV4OnMaintenanceInfoDelegate::CreateLambda([this](const FHiveResultAPI& Result, const TArray<FHiveAuthV4MaintenanceInfo>& AuthV4MaintenanceInfoArray) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("CheckBlacklist FHiveResultAPI:\n%s\n"), *(Result.ToString()));
        
        if (Result.IsSuccess())
        {
            for (const auto& Info: AuthV4MaintenanceInfoArray)
            {
                FString Message = TEXT("FHiveAuthV4MaintenanceInfo\n");
                Message += FString::Printf(TEXT("Title = %s\n"), *(Info.Title));
                Message += FString::Printf(TEXT("Message = %s\n"), *(Info.Message));
                Message += FString::Printf(TEXT("Button = %s\n"), *(Info.Button));
                Message += FString::Printf(TEXT("Action = %s\n"), *(GetNameFromEHiveAuthV4MaintenanceActionType(Info.Action)));
                Message += FString::Printf(TEXT("Url = %s\n"), *(Info.Url));
                Message += FString::Printf(TEXT("StartDate = %s\n"), *(Info.StartDate));
                Message += FString::Printf(TEXT("EndDate = %s\n"), *(Info.EndDate));
                Message += FString::Printf(TEXT("RemainingTime = %s\n"), *(FString::FromInt(Info.RemainingTime)));
                
                Message += TEXT("\n");
                
                AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 , TEXT("%s"), *Message);
            }
        }
        ProcessNeedExit(Result, ELogType::AuthV4);
    }));
    
}

void UPlayerTestView::PlayerSetProviderChangedHandler()
{
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("FHiveAuthV4::SetProviderChangedListener Called\n"));
    
    FHiveAuthV4::SetProviderChangedListener(FHiveAuthV4OnCheckProviderDelegate::CreateLambda([this](const FHiveResultAPI& Result, const FHiveProviderInfo& ProviderInfo) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("SetProviderChangedListener\nFHiveResultAPI : %s\n"), *(Result.ToString()));
        
        if (Result.IsSuccess())
        {
            FString ProviderTypeName = GetNameFromEHiveProviderType(ProviderInfo.ProviderType);
            AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("ProviderType: %s\nProviderUserId: %s"), *ProviderTypeName, *(ProviderInfo.ProviderUserId));
        }
    }));

}
void UPlayerTestView::PlayerShowSignIn()
{
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("FHiveAuthV4::ShowSignIn Called\n"));
    FHiveAuthV4::ShowSignIn(FHiveAuthV4OnSignInDelegate::CreateLambda([this](const FHiveResultAPI& Result, const FHivePlayerInfo& PlayerInfo) {
        if (Result.IsSuccess())
        {
            AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("FHiveAuthV4::ShowSignIn isSuccess\n"));
            this->mCurrentPlayer = PlayerInfo;
            PrintFHivePlayerInfo(PlayerInfo, ELogType::AuthV4);
            RequestPostBoxLogin(); // Request PostBox Login
        } else {
            AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("FHiveAuthV4::ShowSignIn Failed\n"));
            AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("Result = %s"), *(Result.ToString()));
        }
        ProcessNeedExit(Result, ELogType::AuthV4);
    }));
}

void UPlayerTestView::PlayerShowTerms()
{
    FHiveAuthV4::ShowTerms(FHiveAuthV4OnShowTermsDelegate::CreateLambda([this](const FHiveResultAPI& Result) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("ShowTerms Result = %s"), *(Result.ToString()));
    }));
}
                               
void UPlayerTestView::PlayerResetAgreement()
{
    FHiveAuthV4::ResetAgreement();
}

void UPlayerTestView::PlayerShowAdultConfirm()
{
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("FHiveAuthV4::ShowAdultConfirm() Called\n"));
    
    FHiveAuthV4::ShowAdultConfirm(FHiveAuthV4OnAdultConfirmDelegate::CreateLambda([this](const FHiveResultAPI& Result) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("ShowAdultConfirm Result = %s"), *(Result.ToString()));
    }));
}

void UPlayerTestView::PlayerGetProfile(FString playerIdlist)
{
    TArray<FString> TmpArray;
    playerIdlist.ParseIntoArray(TmpArray, TEXT(","));
    
    TArray<int64> PlayerIdArray;
    for (const auto& StrPlayerId: TmpArray)
    {
        PlayerIdArray.Add(FCString::Atoi64(*StrPlayerId));
    }
    
    if (PlayerIdArray.Num() < 1)
    {
        TOptional<FHivePlayerInfo> PlayerInfo = FHiveAuthV4::GetPlayerInfo();
        if (PlayerInfo.IsSet())
        {
            PlayerIdArray.Add(PlayerInfo.GetValue().PlayerId);
        }
    }
    
    FHiveAuthV4::GetProfile(PlayerIdArray, FHiveAuthV4OnGetProfileDelegate::CreateLambda([this](const FHiveResultAPI& Result, const TArray<FHiveProfileInfo> ProfileInfoArray){
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("GetProfile FHiveResultAPI : %s"), *(Result.ToString()));
        
        if (Result.IsSuccess() && ProfileInfoArray.Num() > 0)
        {
            for(const auto& ProfileInfo : ProfileInfoArray)
            {
                FString Message = TEXT("[ProfileInfo]\n");
                Message += FString::Printf(TEXT("PlayerId: %lld\n"), ProfileInfo.PlayerId);
                Message += FString::Printf(TEXT("PlayerName: %s\n"), *(ProfileInfo.PlayerName));
                Message += FString::Printf(TEXT("PlayerImageUrl: %s\n"), *(ProfileInfo.PlayerImageUrl));
                
                AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("%s"), *Message);
            }
        }
    }));
    
}

void UPlayerTestView::PlayerShowProfile(FString playerId)
{
    int64 ipID = -1;
	if(playerId.Len() < 1) // PlayerID를 입력받지 않았다면 현재 플레이어의 플레이어 ID값으로 대체 (GCPHQ-5051)
	{
		ipID = mCurrentPlayer.PlayerId;
	}
	else 
    {
		LexTryParseString(ipID, *playerId);
	}
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("FHiveAuthV4::ShowProfile request player ID : %lld"), ipID);
    
    FHiveAuthV4::ShowProfile(ipID, FHiveAuthV4OnShowProfileDelegate::CreateLambda([this](const FHiveResultAPI& Result){
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("ShowProfile result = %s"), *(Result.ToString()));
    }));
}

bool UPlayerTestView::PlayerIsAutoSignIn()
{
    bool autoSignIn = AuthV4::isAutoSignIn();
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("IsAutoSignIn result : %s\n"), *(LexToString(autoSignIn)));
    
    return autoSignIn;
}

void UPlayerTestView::PlayerGetPlayerInfo()
{
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("FHiveAuthV4::GetPlayerInfo()\n"));

    TOptional<FHivePlayerInfo> PlayerInfo = FHiveAuthV4::GetPlayerInfo();
    if(PlayerInfo.IsSet())
    {
        PrintFHivePlayerInfo(PlayerInfo.GetValue(), ELogType::AuthV4);
    }
}

void UPlayerTestView::PlayerGetParentalConsentInfo()
{
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("FHiveAuthV4::GetParentalConsentInfo()\n"));

    TOptional<FHiveParentalConsentInfo> ParentalConsentInfo = FHiveAuthV4::GetParentalConsentInfo();
    if(ParentalConsentInfo.IsSet())
    {
        FString Message = FString::Printf(TEXT("dateOfBirth: %s\nguardianEmail: %s\n"),
                                      *(ParentalConsentInfo.GetValue().dateOfBirth),
                                      *(ParentalConsentInfo.GetValue().guardianEmail));

        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("%s"), *Message);
    }
}

void UPlayerTestView::PlayerSetEngagementReady(bool isReady)
{
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("SetEngagementReady(%s)"),isReady ? TEXT("TRUE"):TEXT("FALSE"));
    
    TOptional<FHiveResultAPI> Result = FHivePromotion::SetEngagementReady(isReady);
    if (Result.IsSet())
    {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("setEngagementReady result = %s"), *(Result.GetValue().ToString()));
    }
}

void UPlayerTestView::PlayerGetProviderFriendsList()
{
    EHiveProviderType TargetProviderType = EHiveProviderType::FACEBOOK;
    FHiveAuthV4::GetProviderFriendsList(TargetProviderType, FHiveAuthV4OnGetProviderFriendsListDelegate::CreateLambda([this](const FHiveResultAPI& Result,
                                                                                                                         const EHiveProviderType& ProviderType,
                                                                                                                         const ProviderFriendsMap& ProviderUserIdList) {
        
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("GetProviderFriendsList\nFResult = %s\nEHiveProviderType = %s"), *(Result.ToString()), *(GetNameFromEHiveProviderType(ProviderType)));
        
        if (Result.IsSuccess())
        {
            for (const auto& Pair: ProviderUserIdList)
            {
                AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("[%s : %lld]"), *(Pair.Key), Pair.Value);
            }
        }
    }));
}
void UPlayerTestView::PlayerResolveConflict()
{
    FHiveAuthV4::ResolveConflict(FHiveAuthV4OnResolveConflictDelegate::CreateLambda([this](const FHiveResultAPI& Result) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("ResolveConflict\nFResult = %s"), *(Result.ToString()));
    }));
}

//Auth V4 Helper

void UPlayerTestView::PlayerHelperShowSignIn()
{
    FHiveAuthV4::ShowSignIn(FHiveAuthV4OnSignInDelegate::CreateLambda([this](const FHiveResultAPI& Result, const FHivePlayerInfo& PlayerInfo) {
        
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4Helper ,TEXT("ShowSignIn\nFResult = %s"), *(Result.ToString()));
        if (Result.IsSuccess())
        {
            this->mCurrentPlayer = PlayerInfo;
            PrintFHivePlayerInfo(PlayerInfo, ELogType::AuthV4Helper);
            RequestPostBoxLogin(); // Request PostBox Login
        }
        ProcessNeedExit(Result, ELogType::AuthV4Helper);
    }));
}

void UPlayerTestView::PlayerHelperSignIn()
{
    FHiveAuthV4::Helper::SignIn(FHiveAuthV4HelperDelegate::CreateLambda([this](const FHiveResultAPI& Result, const TOptional<FHivePlayerInfo>& PlayerInfo) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4Helper ,TEXT("Helper::SignIn Result = %s"), *(Result.ToString()));
        
        if (PlayerInfo.IsSet())
        {
            PrintFHivePlayerInfo(PlayerInfo.GetValue(), ELogType::AuthV4Helper);
        }
        
        switch (Result.Code) {
            case FHiveResultAPI::ECode::Success: {
                // 로그인 성공
                if (PlayerInfo.IsSet())
                {
                    this->mCurrentPlayer = PlayerInfo.GetValue();
                    RequestPostBoxLogin(); // Request PostBox Login
                }
                else
                {
                    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4Helper ,TEXT("PlayerInfo is not set."));
                }
            }
                break;
            default:
                break;
                
        }
        
        ProcessNeedExit(Result, ELogType::AuthV4Helper);
    }));
}


void UPlayerTestView::PlayerHelperSignOut()
{
    
    FHiveAuthV4::Helper::SignOut(FHiveAuthV4HelperDelegate::CreateLambda([this](const FHiveResultAPI& Result, const TOptional<FHivePlayerInfo>& PlayerInfo) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4Helper ,TEXT("Helper::SignOut Result = %s"), *(Result.ToString()));
        
        if(PlayerInfo.IsSet())
        {
            PrintFHivePlayerInfo(PlayerInfo.GetValue(), ELogType::AuthV4);
        }
    }));
}
    
    
void UPlayerTestView::PlayerHelperDelete()
{
    FHiveAuthV4::Helper::PlayerDelete(FHiveAuthV4HelperDelegate::CreateLambda([this](const FHiveResultAPI& Result, const TOptional<FHivePlayerInfo>& PlayerInfo) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4Helper ,TEXT("Helper::PlayerDelete Result = %s"), *(Result.ToString()));
        
        if(PlayerInfo.IsSet())
        {
            PrintFHivePlayerInfo(PlayerInfo.GetValue(), ELogType::AuthV4);
        }
    }));
}

void UPlayerTestView::PlayerHelperConnect(FString providerType)
{
    EHiveProviderType ProviderType = GetEHiveProviderTypeFromName(providerType);
    FHiveAuthV4::Helper::Connect(ProviderType, FHiveAuthV4HelperDelegate::CreateLambda([this](const FHiveResultAPI& Result, const TOptional<FHivePlayerInfo>& PlayerInfo) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4Helper ,TEXT("Helper::Connect Result = %s"), *(Result.ToString()));
        
        if (PlayerInfo.IsSet())
        {
            PrintFHivePlayerInfo(PlayerInfo.GetValue(), ELogType::AuthV4Helper);
        }
        
        if(Result.ErrorCode == FHiveResultAPI::EErrorCode::CONFLICT_PLAYER)
        {
            if (PlayerInfo.IsSet())
            {
                this->mConflictPlayer = PlayerInfo.GetValue();
            }
        }
        
        ProcessNeedExit(Result, ELogType::AuthV4Helper);
    }));
}

void UPlayerTestView::PlayerHelperDisconnect(FString providerType)
{
    EHiveProviderType ProviderType = GetEHiveProviderTypeFromName(providerType);
    FHiveAuthV4::Helper::Disconnect(ProviderType, FHiveAuthV4HelperDelegate::CreateLambda([this](const FHiveResultAPI& Result, const TOptional<FHivePlayerInfo>& PlayerInfo) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4Helper ,TEXT("Helper::Disconnect Result = %s"), *(Result.ToString()));
        
        if(PlayerInfo.IsSet())
        {
            PrintFHivePlayerInfo(PlayerInfo.GetValue(), ELogType::AuthV4Helper);
        }
    }));
}


void UPlayerTestView::PlayerHelperGetIDPList()
{
    TArray<EHiveProviderType> IDPList = FHiveAuthV4::Helper::GetIDPList();
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4Helper ,TEXT("Helper::GetIDPList()\n"));
    for(const auto& IDP : IDPList)
    {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4Helper ,TEXT("ProviderType = %s\n"), *(GetNameFromEHiveProviderType(IDP)));
    }
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4Helper ,TEXT("\n"));
}


void UPlayerTestView::PlayerHelperSyncAccount(FString providerType)
{
    EHiveProviderType ProviderType = GetEHiveProviderTypeFromName(providerType);
    FHiveAuthV4::Helper::SyncAccount(ProviderType, FHiveAuthV4HelperDelegate::CreateLambda([this](const FHiveResultAPI& Result, const TOptional<FHivePlayerInfo>& PlayerInfo) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4Helper ,TEXT("Helper::SyncAccount Result = %s"), *(Result.ToString()));
        
        if (PlayerInfo.IsSet())
        {
            PrintFHivePlayerInfo(PlayerInfo.GetValue(), ELogType::AuthV4Helper);
        }
        
        if(Result.ErrorCode == FHiveResultAPI::EErrorCode::CONFLICT_PLAYER)
        {
            if (PlayerInfo.IsSet())
            {
                this->mConflictPlayer = PlayerInfo.GetValue();
            }
        }
    }));
}

void UPlayerTestView::PlayerHelperShowAchievements()
{
    FHiveAuthV4::Helper::ShowAchievements(FHiveAuthV4HelperDelegate::CreateLambda([this](const FHiveResultAPI& Result, const TOptional<FHivePlayerInfo>& PlayerInfo) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4Helper ,TEXT("Helper::ShowAchievements Result = %s"), *(Result.ToString()));
        
        if(PlayerInfo.IsSet())
        {
            PrintFHivePlayerInfo(PlayerInfo.GetValue(), ELogType::AuthV4Helper);
        }
    }));
}

void UPlayerTestView::PlayerHelperShowLeaderboard()
{
    FHiveAuthV4::Helper::ShowLeaderboard(FHiveAuthV4HelperDelegate::CreateLambda([this](const FHiveResultAPI& Result, const TOptional<FHivePlayerInfo>& PlayerInfo) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4Helper ,TEXT("Helper::ShowLeaderboard Result = %s"), *(Result.ToString()));
        
        if(PlayerInfo.IsSet())
        {
            PrintFHivePlayerInfo(PlayerInfo.GetValue(), ELogType::AuthV4Helper);
        }
    }));
}

void UPlayerTestView::PlayerHelperShowConflict()
{
    FHiveAuthV4::Helper::ShowConflict(FHiveAuthV4HelperDelegate::CreateLambda([this](const FHiveResultAPI& Result, const TOptional<FHivePlayerInfo>& PlayerInfo) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4Helper ,TEXT("Helper::ShowConflict Result = %s"), *(Result.ToString()));
        
        if(PlayerInfo.IsSet())
        {
            this->mCurrentPlayer = PlayerInfo.GetValue();
            PrintFHivePlayerInfo(PlayerInfo.GetValue(), ELogType::AuthV4Helper);
            RequestPostBoxLogin(); // Request PostBox Login
        }
    }));
}

void UPlayerTestView::PlayerHelperShowConflict2()
{
    FHiveConflictSingleViewData ViewData = FHiveConflictSingleViewData(mConflictPlayer.PlayerId);
    ViewData.SetData(TEXT("Gold"), TEXT("500000"));
    ViewData.SetData(TEXT("Gem"), TEXT("5000"));
    
    FHiveAuthV4::Helper::ShowConflict(ViewData, FHiveAuthV4HelperDelegate::CreateLambda([this](const FHiveResultAPI& Result, const TOptional<FHivePlayerInfo>& PlayerInfo) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4Helper ,TEXT("Helper::ShowConflict2 Result = %s"), *(Result.ToString()));
        
        if(PlayerInfo.IsSet())
        {
            this->mCurrentPlayer = PlayerInfo.GetValue();
            PrintFHivePlayerInfo(PlayerInfo.GetValue(), ELogType::AuthV4Helper);
            RequestPostBoxLogin(); // Request PostBox Login
        }
    }));
}

void UPlayerTestView::PlayerHelperSwitchAccount()
{
    FHiveAuthV4::Helper::SwitchAccount(FHiveAuthV4HelperDelegate::CreateLambda([this](const FHiveResultAPI& Result, const TOptional<FHivePlayerInfo>& PlayerInfo) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4Helper ,TEXT("Helper::SwitchAccount Result = %s"), *(Result.ToString()));
        
        if(PlayerInfo.IsSet())
        {
            PrintFHivePlayerInfo(PlayerInfo.GetValue(), ELogType::AuthV4Helper);
        }
    }));
}

void UPlayerTestView::PlayerHelperResolveConflict()
{
    FHiveAuthV4::Helper::ResolveConflict(FHiveAuthV4HelperDelegate::CreateLambda([this](const FHiveResultAPI& Result, const TOptional<FHivePlayerInfo>& PlayerInfo) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4Helper ,TEXT("Helper::ResolveConflict Result = %s"), *(Result.ToString()));
        
        if(PlayerInfo.IsSet())
        {
            PrintFHivePlayerInfo(PlayerInfo.GetValue(), ELogType::AuthV4Helper);
        }
    }));
}

void UPlayerTestView::PlayerHelperShowGameCenterLoginCancelDialog()
{
    FHiveAuthV4::Helper::ShowGameCenterLoginCancelDialog(FHiveAuthV4HelperOnDialogDismissDelegate::CreateLambda([this](bool IsDismiss){
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4,
                                                                         TEXT("Helper::ShowGameCenterLoginCancelDialog\nIsDismiss = %s"),
                                                                         IsDismiss ? TEXT("TRUE") : TEXT("FALSE"));
    }));
}

void UPlayerTestView::PlayerShowGameCenterLoginCancelDialog()
{
    FHiveAuthV4::ShowGameCenterLoginCancelDialog(FHiveAuthV4OnAuthV4DialogDismissDelegate::CreateLambda([this](bool IsDismiss){
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4,
                                                                         TEXT("ShowGameCenterLoginCancelDialog\nIsDismiss = %s"),
                                                                         IsDismiss ? TEXT("TRUE") : TEXT("FALSE"));
    }));
}


void UPlayerTestView::PlayerRequestPermissionViewData()
{
	AuthV4::requestPermissionViewData([=](ResultAPI const & result,PermissionViewData const & permissionData){
		
		
		FString permissionStringData;
		
		//	permissionData.permissions
		//	permissionData.commons
		
		permissionStringData.Append(TEXT("\n===PERMISSIONS===\n"));
		for(auto permission = permissionData.permissions.begin() ; permission != permissionData.permissions.end(); permission++)
		{
			PermissionViewUnitData permissionObj = (PermissionViewUnitData)*permission;
			permissionStringData.Append(FUTF8ToTCHAR(permissionObj.title.c_str()).Get());
			permissionStringData.Append(TEXT("\n"));
			permissionStringData.Append(FUTF8ToTCHAR(permissionObj.contents.c_str()).Get());
			permissionStringData.Append(TEXT("\n"));
			permissionStringData.Append(FUTF8ToTCHAR(permissionObj.nativePermissionName.c_str()).Get());
			permissionStringData.Append(TEXT("\n"));
		}
		permissionStringData.Append(TEXT("\n===COMMONS===\n"));
		for(auto permission = permissionData.commons.begin() ; permission != permissionData.commons.end(); permission++)
		{
			PermissionViewUnitData permissionObj = (PermissionViewUnitData)*permission;
			permissionStringData.Append(FUTF8ToTCHAR(permissionObj.title.c_str()).Get());
			permissionStringData.Append(TEXT("\n"));
			permissionStringData.Append(FUTF8ToTCHAR(permissionObj.contents.c_str()).Get());
			permissionStringData.Append(TEXT("\n"));
			permissionStringData.Append(FUTF8ToTCHAR(permissionObj.nativePermissionName.c_str()).Get());
			permissionStringData.Append(TEXT("\n"));
		}
		permissionStringData.Append(TEXT("\n===DESCRIPTION===\n"));
		permissionStringData.Append(FUTF8ToTCHAR(permissionData.contents.c_str()).Get());
		permissionStringData.Append(TEXT("\n"));

		AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("RequestPermissionViewData called\n%s\n\n"), *permissionStringData);

	});
}


bool UPlayerTestView::PlayerGetAgeGateU13()
{
    bool bAgeGateU13 = FHiveAuthV4::GetAgeGateU13();
	AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4,
                                                                     TEXT("GetAgeGateU13 : %s\n"),
                                                                     bAgeGateU13 ? TEXT("TRUE") : TEXT("FALSE"));
	return bAgeGateU13;
}



void UPlayerTestView::PlayerRequestPlatformHelperPermission(FString permissions)
{
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("=== RequestUserPermissions() ==="));
    
    TArray<FString> Requests;
    permissions.ParseIntoArrayLines(Requests, true);
    
    for (const auto& Request : Requests)
    {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("Request= %s"), *Request);
    }
    
    FHivePlatformHelper::RequestUserPermissions(Requests, FHivePlatformHelperOnUserPermissionsDelegate::CreateLambda([this](const FHiveResultAPI& Result, const TArray<FString>& GrantedRequests, const TArray<FString>& DeniedRequests) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("%s"), *(Result.ToString()));
        
        if (GrantedRequests.Num() > 0)
        {
            AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("<Granted Requests>"));
            for (const auto& Request : GrantedRequests)
            {
                AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("Request= %s"), *Request);
            }
        }
        
        if (DeniedRequests.Num() > 0)
        {
            AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("<Denied Requests>"));
            for (const auto& Request : DeniedRequests)
            {
                AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("Request= %s"), *Request);
            }
        }
    }));
    
}

void UPlayerTestView::PlayerShowInquiry() {
    FHiveAuthV4::ShowInquiry(FHiveAuthV4OnShowInquiryDelegate::CreateLambda([this](const FHiveResultAPI& Result) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("ShowInquiary result = %s"), *(Result.ToString()));
    }));
}

void UPlayerTestView::PlayerShowMyInquiry() {
    FHiveAuthV4::ShowMyInquiry(FHiveAuthV4OnShowInquiryDelegate::CreateLambda([this](const FHiveResultAPI& Result) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("ShowMyInquiary result = %s"), *(Result.ToString()));
    }));
}

void UPlayerTestView::PlayerShowChatbotInquiry() {
    // 추가 정보를 담을 FJsonObject 생성
    TSharedPtr<FJsonObject> AdditionalInfoJson = MakeShareable(new FJsonObject);
    AdditionalInfoJson->SetStringField(TEXT("evt_code"), TEXT("1000"));
    AdditionalInfoJson->SetStringField(TEXT("p1"), TEXT("random"));

    // FJsonObject를 문자열로 직렬화
    FString StrAdditionalInfo;
    TSharedRef<TJsonWriter<>> JsonWriter = TJsonWriterFactory<>::Create(&StrAdditionalInfo);
    FJsonSerializer::Serialize(AdditionalInfoJson.ToSharedRef(), JsonWriter);

    FHiveAuthV4::ShowChatbotInquiry(StrAdditionalInfo, FHiveAuthV4OnShowInquiryDelegate::CreateLambda([this](const FHiveResultAPI& Result) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("ShowChatbotInquiry result = %s"), *(Result.ToString()));
    }));
}

void UPlayerTestView::PlayerShowDeviceManagement() {
    FHiveAuthV4::ShowDeviceManagement(FHiveAuthV4OnShowDeviceManagementDelegate::CreateLambda([this](const FHiveResultAPI& Result) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("ShowDeviceManagement\nFResult = %s"), *(Result.ToString()));
    }));
}

void UPlayerTestView::PlayerSignInWithStoredPlayerId(FString playerId, bool useAutoSignIn) {
    int64 PlayerId;
    LexFromString(PlayerId, *playerId);

    FHiveAuthV4::SignInWithStoredPlayerId(PlayerId, useAutoSignIn, FHiveAuthV4OnSignInDelegate::CreateLambda([this](const FHiveResultAPI& Result, const FHivePlayerInfo& PlayerInfo) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("SignIn result : %s"), *(Result.ToString()));
        if (Result.IsSuccess())
        {
            this->mCurrentPlayer = PlayerInfo;
            PrintFHivePlayerInfo(PlayerInfo, ELogType::AuthV4);
            RequestPostBoxLogin(); // Request PostBox Login
        }
        ProcessNeedExit(Result, ELogType::AuthV4);
    }));
}

void UPlayerTestView::PlayerStoreCurrentPlayerId() {
    FHiveAuthV4::StoreCurrentPlayerId(FHiveAuthV4OnStoreCurrentPlayerIdDelegate::CreateLambda([this](const FHiveResultAPI& Result) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("StoreCurrentPlayerId Result: %s"), *(Result.ToString()));
    }));
}

void UPlayerTestView::PlayerGetStoredPlayerIdList() {
    FHiveAuthV4::GetStoredPlayerIdList(FHiveAuthV4OnGetStoredPlayerIdListDelegate::CreateLambda(
        [this](const FHiveResultAPI& Result, const TArray<int64>& PlayerIdList) {
            AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("GetStoredPlayerIdList Result: %s"), *(Result.ToString()));
            if (Result.IsSuccess() && PlayerIdList.Num() > 0) {
                FString IdListStr;
                for (const auto& Id : PlayerIdList) {
                    IdListStr += FString::Printf(TEXT("%lld "), Id);
                }

                AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("PlayerIdList: %s"), *IdListStr);

                mPlayerIdList = PlayerIdList;

                if (AHIVESDKV4TesterGameMode::getGameModeInstance()->OnUpdateGetStoredPlayerIdList.IsBound())
                {
                    AHIVESDKV4TesterGameMode::getGameModeInstance()->OnUpdateGetStoredPlayerIdList.Broadcast();
                }
            }
        }
    ));

}

void UPlayerTestView::PlayerDeleteStoredPlayerId(FString playerId) {
    int64 PlayerId;
    LexFromString(PlayerId, *playerId);
    FHiveAuthV4::DeleteStoredPlayerId(PlayerId, FHiveAuthV4OnDeleteStoredPlayerIdDelegate::CreateLambda([this, PlayerId](const FHiveResultAPI& Result) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("DeleteStoredPlayerId Result: %s"), *(Result.ToString()));

        if (Result.IsSuccess())
        {
            // mPlayerIdList에서 PlayerId 제거
            mPlayerIdList.Remove(PlayerId);

            // 화면 갱신
            if (AHIVESDKV4TesterGameMode::getGameModeInstance()->OnUpdateGetStoredPlayerIdList.IsBound())
            {
                AHIVESDKV4TesterGameMode::getGameModeInstance()->OnUpdateGetStoredPlayerIdList.Broadcast();
            }
        }

    }));
}

void UPlayerTestView::PlayerDeleteStoredPlayerIdAll() {
    // 삭제할 PlayerId 리스트를 복사
    TArray<int64> PlayerIdsToDelete = mPlayerIdList;

    // 내부적으로 순차 삭제를 위한 람다 정의
    TSharedPtr<TFunction<void(int32)>> DeleteNext = MakeShared<TFunction<void(int32)>>();

    *DeleteNext = [this, PlayerIdsToDelete, DeleteNext](int32 Index)
        {
            if (Index >= PlayerIdsToDelete.Num())
            {
                // 모두 삭제 완료 후 화면 갱신
                if (AHIVESDKV4TesterGameMode::getGameModeInstance()->OnUpdateGetStoredPlayerIdList.IsBound())
                {
                    AHIVESDKV4TesterGameMode::getGameModeInstance()->OnUpdateGetStoredPlayerIdList.Broadcast();
                }
                return;
            }

            int64 PlayerId = PlayerIdsToDelete[Index];
            FHiveAuthV4::DeleteStoredPlayerId(PlayerId, FHiveAuthV4OnDeleteStoredPlayerIdDelegate::CreateLambda(
                [this, PlayerId, Index, DeleteNext](const FHiveResultAPI& Result)
                {
                    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(
                        ELogType::AuthV4, TEXT("DeleteStoredPlayerId Result: %s"), *(Result.ToString()));

                    if (Result.IsSuccess())
                    {
                        mPlayerIdList.Remove(PlayerId);
                    }

                    // 다음 삭제 진행
                    (*DeleteNext)(Index + 1);
                }
            ));
        };

    // 삭제 시작
    if (PlayerIdsToDelete.Num() > 0)
    {
        (*DeleteNext)(0);
    }
    else
    {
        if (AHIVESDKV4TesterGameMode::getGameModeInstance()->OnUpdateGetStoredPlayerIdList.IsBound())
        {
            AHIVESDKV4TesterGameMode::getGameModeInstance()->OnUpdateGetStoredPlayerIdList.Broadcast();
        }
    }
}

void UPlayerTestView::PlayerGetHashedDi() {
    FHiveAuthV4::GetHashedDi(FHiveAuthV4OnGetHashedDiDelegate::CreateLambda([this](const FHiveResultAPI& Result, const FString& HashedDi) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("GetHashedDi Result: %s, HashedDi: %s"), *(Result.ToString()), *HashedDi);
        }));
}

void UPlayerTestView::PlayerShowIdentityVerification() {
    FHiveAuthV4::ShowIdentityVerification(FHiveAuthV4OnIdentityVerificationDelegate::CreateLambda([this](const FHiveResultAPI& Result, const FHiveIdentity& Identity) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("ShowIdentityVerification Result: %s, Identity: %s"), *(Result.ToString()), *(GetNameFromFHiveIdentity(Identity)));
    }));
}

void UPlayerTestView::GetAgeRange() {
    FHiveAuthV4::GetAgeRange(FHiveAuthV4OnGetAgeRangeDelegate::CreateLambda([this](const FHiveResultAPI& Result, const FHiveAuthV4AgeRange& AgeRange) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("GetAgeRange Result: %s"), *(Result.ToString()));
        PrintAgeRange(AgeRange);
    }));
}

void UPlayerTestView::ShowAgeRangeUpdatePermission(FString Description) {
    FHiveAuthV4::ShowAgeRangeUpdatePermission(Description, FHiveAuthV4OnShowAgeRangeUpdatePermissionDelegate::CreateLambda([this](const FHiveResultAPI& Result, const FHiveAuthV4AgeRange& AgeRange) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("ShowAgeRangeUpdatePermission Result: %s"), *(Result.ToString()));
        PrintAgeRange(AgeRange);
    }));
}

void PrintAgeRange(const FHiveAuthV4AgeRange& AgeRange)
{
    FString UserStatusString;

    switch (AgeRange.UserStatus)
    {
    case EHiveUserStatus::VERIFIED:
        UserStatusString = TEXT("VERIFIED");
        break;

    case EHiveUserStatus::SUPERVISED:
        UserStatusString = TEXT("SUPERVISED");
        break;

    case EHiveUserStatus::SUPERVISED_APPROVAL_PENDING:
        UserStatusString = TEXT("SUPERVISED_APPROVAL_PENDING");
        break;

    case EHiveUserStatus::SUPERVISED_APPROVAL_DENIED:
        UserStatusString = TEXT("SUPERVISED_APPROVAL_DENIED");
        break;

    case EHiveUserStatus::REQUIRED:
        UserStatusString = TEXT("REQUIRED");
        break;

    case EHiveUserStatus::UNKNOWN:
    default:
        UserStatusString = TEXT("UNKNOWN");
        break;
    }

    const FString Message = FString::Printf(
        TEXT("UserStatus: %s\nAgeLower: %d\nAgeUpper: %d\nMostRecentApprovalDate: %s\nAgeRangeId: %s\n"),
        *UserStatusString,
        AgeRange.AgeLower,
        AgeRange.AgeUpper,
        *AgeRange.MostRecentApprovalDate,
        *AgeRange.AgeRangeId
    );

    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(
        ELogType::AuthV4,
        TEXT("AgeRange: %s"),
        *Message
    );
}

void UPlayerTestView::PlayerCheckAgeGate(bool UseCloseButton) {
    FHiveAuthV4::CheckAgeGate(UseCloseButton, FHiveAuthV4OnCheckAgeGateDelegate::CreateLambda([this](const FHiveResultAPI& Result, const FString& DateOfBirth) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("CheckAgeGate Result: %s, DateOfBirth: %s"), *(Result.ToString()), *DateOfBirth);
        }));
}

FString UPlayerTestView::GetHiveCountry() {
    return FHiveConfiguration::GetHiveCountry();
}

void UPlayerTestView::HIVEConfigurationGetMetaData(FString key, FString forceReload) {
    bool bForceReload = forceReload.ToBool();
    
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("GetMetaData(%s, %s)"), *key, *forceReload);
    FHiveConfiguration::GetMetaData(key, bForceReload, FHiveConfigurationOnMetaDataDelegate::CreateLambda([this](const FHiveResultAPI& Result, const FString& Value) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4 ,TEXT("%sValue = %s"), *(Result.ToString()), *Value);
    }));
}

void UPlayerTestView::PlayerSetAutoLoginEnabled(bool enabled) {
    FHiveAuthV4::SetAutoLoginEnabled(enabled, FHiveAuthV4OnSetAutoLoginEnabledDelegate::CreateLambda([this](const FHiveResultAPI& Result, bool IsEnabled) {
        bAutoLoginEnabled = IsEnabled;
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::AuthV4, TEXT("SetAutoLoginEnabled Result: %s, IsEnabled: %s"), *(Result.ToString()), IsEnabled ? TEXT("TRUE") : TEXT("FALSE"));

        if (AHIVESDKV4TesterGameMode::getGameModeInstance()->OnUpdateSetAutoLoginEnabled.IsBound())
        {
            AHIVESDKV4TesterGameMode::getGameModeInstance()->OnUpdateSetAutoLoginEnabled.Broadcast();
        }
    }));
}

void UPlayerTestView::RequestPostBoxLogin()
{
    if (AHIVESDKV4TesterGameMode::getGameModeInstance()->OnRequestPostBoxLogin.IsBound())
    {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->OnRequestPostBoxLogin.Broadcast();
    }
}