/* Copyright © 2024 Com2uS Platform Corp. All Rights Reserved. */

#include "TestAppGameNetwork.h"
#include "HIVESDKV4Tester.h"
#include "CoreMinimal.h"

#include "HIVESDKV4TesterGameMode.h"
#include "HiveTestUtils.h"
#include "TestAppUserGameInstance.h"

namespace
{
    FString GetGameLanguageFromSavedFile()
    {
        FString ReadText;
        FString SavedConfigFile = FPaths::Combine(FPaths::ProjectDir(), TEXT("HIVESDKV4ConfigSaved.txt"));

        if (FFileHelper::LoadFileToString(ReadText, *SavedConfigFile))
        {
            TSharedPtr<FJsonObject> JsonConfigObj;
            TSharedRef<TJsonReader<>> Reader = TJsonReaderFactory<>::Create(ReadText);

            if (FJsonSerializer::Deserialize(Reader, JsonConfigObj) && JsonConfigObj->HasField(TEXT("gameLanguage")))
            {
                return JsonConfigObj->GetStringField(TEXT("gameLanguage"));
            }
        }

        UE_LOG(LogTemp, Warning, TEXT("TestAppGameNetwork::GetGameLanguageFromSavedFile failed"));
        return TEXT("");
    }
}

TMap<FString, TestAppGameNetwork::FGameListener> listenerMap;

void TestAppGameNetwork::login(FGameListener listener)
{
    TSharedPtr<FJsonObject> params = MakeShareable(new FJsonObject);
    params->SetStringField(TEXT("world"), FHiveConfiguration::GetServerId());
    params->SetStringField(TEXT("uid"), "0");
    params->SetStringField(TEXT("appid"), FHiveConfiguration::GetAppId());
    params->SetStringField(TEXT("country"), FHiveConfiguration::GetHiveCountry());
    params->SetStringField(TEXT("language"), GetGameLanguageFromSavedFile());
    params->SetStringField(TEXT("appver"), FHiveConfiguration::GetAppId());

    TOptional<FHivePlayerInfo> PlayerInfo = FHiveAuthV4::GetPlayerInfo();
    if (PlayerInfo.IsSet())
    {
        FString strPlayerId = FString::Printf(TEXT("%lld"), PlayerInfo.GetValue().PlayerId);
        params->SetStringField(TEXT("vid"), strPlayerId);
        params->SetStringField(TEXT("did"), PlayerInfo.GetValue().Did);
    }
    else
    {
        UE_LOG(LogTemp, Warning, TEXT("TestAppGameNetwork::login PlayerInfo is not set"));
        params->SetStringField(TEXT("vid"), TEXT(""));
        params->SetStringField(TEXT("did"), TEXT(""));
    }
    
    TestAppGameNetwork::request(GetUrl(TEXT("/account/login")), params, listener);
}

void TestAppGameNetwork::customAuth(FString idpId, FString idpUserId, FString authType, FGameListener listener)
{
    TSharedPtr<FJsonObject> params = MakeShareable(new FJsonObject);
    params->SetStringField(TEXT("appid"), FHiveConfiguration::GetAppId());
    params->SetStringField(TEXT("idp_id"), idpId);
    params->SetStringField(TEXT("idp_user_id"), idpUserId);
    params->SetStringField(TEXT("auth_type"), authType);

    //When type is Connect, Add vid. playerid.
    if (authType.Equals("CONNECT"))
    {
        TOptional<FHivePlayerInfo> PlayerInfo = FHiveAuthV4::GetPlayerInfo();
        if (PlayerInfo.IsSet())
        {
            FString strPlayerId = FString::Printf(TEXT("%lld"), PlayerInfo.GetValue().PlayerId);
            params->SetStringField(TEXT("vid"), strPlayerId);
        }
    }

    TestAppGameNetwork::request(GetUrl(TEXT("/account/custom_auth")), params, listener);
}

void TestAppGameNetwork::posts(FGameListener listener)
{
    TSharedPtr<FJsonObject> params = MakeShareable(new FJsonObject);
    AHIVESDKV4TesterGameMode* GameMode = AHIVESDKV4TesterGameMode::getGameModeInstance();    
    UTestAppUserGameInstance* UserGI = UTestAppUserGameInstance::GetInstance(GameMode);
    if (UserGI)
    {
        params->SetStringField(TEXT("session_token"), UserGI->sessionToken);
    }
    params->SetStringField(TEXT("world"), FHiveConfiguration::GetServerId());
    
    TOptional<FHivePlayerInfo> PlayerInfo = FHiveAuthV4::GetPlayerInfo();
    if (PlayerInfo.IsSet())
    {
        FString strPlayerId = FString::Printf(TEXT("%lld"), PlayerInfo.GetValue().PlayerId);
        params->SetStringField(TEXT("vid"), strPlayerId);
    }
    else
    {
        UE_LOG(LogTemp, Warning, TEXT("TestAppGameNetwork::posts PlayerInfo is not set"));
        params->SetStringField(TEXT("vid"), TEXT(""));
    }

    TestAppGameNetwork::request(GetUrl(TEXT("/postbox/posts")), params, listener);
}

void TestAppGameNetwork::postReceive(FGameListener listener, TArray<UPostBoxItemConsumable*> posts)
{
    TSharedPtr<FJsonObject> params = MakeShareable(new FJsonObject);
    AHIVESDKV4TesterGameMode* GameMode = AHIVESDKV4TesterGameMode::getGameModeInstance();    
    UTestAppUserGameInstance* UserGI = UTestAppUserGameInstance::GetInstance(GameMode);
    if (UserGI)
    {
        params->SetStringField(TEXT("session_token"), UserGI->sessionToken);
    }   
    params->SetStringField(TEXT("world"), FHiveConfiguration::GetServerId());

    TOptional<FHivePlayerInfo> PlayerInfo = FHiveAuthV4::GetPlayerInfo();
    if (PlayerInfo.IsSet())
    {
        FString strPlayerId = FString::Printf(TEXT("%lld"), PlayerInfo.GetValue().PlayerId);
        params->SetStringField(TEXT("vid"), strPlayerId);
    }
    else
    {
        UE_LOG(LogTemp, Warning, TEXT("TestAppGameNetwork::postReceive PlayerInfo is not set"));
        params->SetStringField(TEXT("vid"), TEXT(""));
    }

    if (posts.Num() > 0)
    {
        TArray<TSharedPtr<FJsonValue>> postIDList;
        for (UPostBoxItemConsumable* post : posts)
        {
            TSharedPtr<FJsonValue> Value = MakeShareable(new FJsonValueString(*(post->postInfo.postId)));
            postIDList.Add(Value);
        }
        params->SetArrayField(TEXT("post_id_list"), postIDList);
    }

    TestAppGameNetwork::request(GetUrl(TEXT("/postbox/receive")), params, listener);
}

void TestAppGameNetwork::verifyReceipt(FGameListener listener, FString bypassReceipt) {
	verifyReceipt(listener, bypassReceipt, TEXT("consumable"));	//	default "consumable"
}

void TestAppGameNetwork::verifyReceipt(FGameListener listener, FString bypassReceipt, FString itemType)
{    
    TSharedPtr<FJsonObject> params = MakeShareable(new FJsonObject);
    AHIVESDKV4TesterGameMode* GameMode = AHIVESDKV4TesterGameMode::getGameModeInstance();    
    UTestAppUserGameInstance* UserGI = UTestAppUserGameInstance::GetInstance(GameMode);
    if (UserGI)
    {
        params->SetStringField(TEXT("session_token"), UserGI->sessionToken);
    }   
    params->SetStringField(TEXT("world"), FHiveConfiguration::GetServerId());
    params->SetStringField(TEXT("item_type"), itemType); // "consumable" or "subscription"
    params->SetStringField(TEXT("purchase_bypass_info"), bypassReceipt);

    TOptional<FHivePlayerInfo> PlayerInfo = FHiveAuthV4::GetPlayerInfo();
    if (PlayerInfo.IsSet())
    {
        FString strPlayerId = FString::Printf(TEXT("%lld"), PlayerInfo.GetValue().PlayerId);
        params->SetStringField(TEXT("vid"), strPlayerId);
    }
    else
    {
        UE_LOG(LogTemp, Warning, TEXT("TestAppGameNetwork::verifyReceipt PlayerInfo is not set"));
        params->SetStringField(TEXT("vid"), TEXT(""));
    }
    
    TestAppGameNetwork::request(GetUrl(TEXT("/hive/iap/verify")), params, TEXT("text/plain")/* Server Error Disallowed Key Characters */, listener);
}

FString TestAppGameNetwork::GetUrl(FString Path)
{
    FString BaseUrl;

    if (HiveTestUtils::ConfigurationIsTestZone()) {
        BaseUrl = TEXT("https://test-misample.com2us.net");
    }
    else {
        switch (FHiveConfiguration::GetZone())
        {
            case EHiveZoneType::REAL:
                BaseUrl = TEXT("https://misample.com2us.net");
                break;
            case EHiveZoneType::SANDBOX:
                BaseUrl = TEXT("https://sandbox-misample.com2us.net");
                break;
            default:
                BaseUrl = TEXT("");
                break;
        }
    }

    if (BaseUrl.IsEmpty() || Path.IsEmpty())
    {
        UE_LOG(LogTemp, Warning, TEXT("TestAppGameNetwork::GetUrl is empty.\nBaseUrl = %s\nPath = %s"), *BaseUrl, *Path);
    }

    return BaseUrl + Path;
}

void TestAppGameNetwork::request(FString url, TSharedPtr<FJsonObject> postData, FGameListener listener)
{
    TestAppGameNetwork::request(url, postData, TEXT("application/x-www-form-urlencoded"), listener);
}

void TestAppGameNetwork::request(FString url, TSharedPtr<FJsonObject> postData, FString contentType, FGameListener listener)
{
    FString SerializedPostData;
    TSharedRef< TJsonWriter<TCHAR, TCondensedJsonPrintPolicy< TCHAR >> > Writer = TJsonWriterFactory<TCHAR, TCondensedJsonPrintPolicy< TCHAR >>::Create(&SerializedPostData);
    if (FJsonSerializer::Serialize(postData.ToSharedRef(), Writer))
    {
        TSharedRef<IHttpRequest, ESPMode::ThreadSafe> request = FHttpModule::Get().CreateRequest();
        request->SetHeader(TEXT("Content-Type"), contentType);
        request->SetURL(url);
        request->SetVerb(TEXT("POST"));
        request->SetContentAsString(SerializedPostData);
        request->OnProcessRequestComplete().BindStatic(&TestAppGameNetwork::connectionDidFinishLoading);
        
#if ENGINE_MAJOR_VERSION == 5
    #if ENGINE_MINOR_VERSION >= 5
        // UE 5.5 이상: SetTimeout 사용
        request->SetTimeout(300.0f);
    #else
        // UE 5.4 이하: SetHttpTimeout 사용
        FHttpModule::Get().SetHttpTimeout(300);
    #endif
#else
    // UE 4.x: SetHttpTimeout 사용
    FHttpModule::Get().SetHttpTimeout(300);
#endif
        
        listenerMap.Add(url, listener);
        if (!request->ProcessRequest())
        {
            UE_LOG(LogTemp, Warning, TEXT("TestAppGameNetwork::request() ProcessRequest() failed"));
        }
    }
    else
    {
        UE_LOG(LogTemp, Warning, TEXT("TestAppGameNetwork::request() postData serialization failed"));
        ITestAppGameNetworkResponse reponse;
        listener.ExecuteIfBound(false, reponse);
    }
}

void TestAppGameNetwork::connectionDidFinishLoading(FHttpRequestPtr Request, FHttpResponsePtr Response, bool bWasSuccessful)
{
	TestAppGameNetwork::FGameListener * responseListner = listenerMap.Find(Request->GetURL());

    ITestAppGameNetworkResponse rtnConnectResponse;
    if (bWasSuccessful &&
        Response.IsValid())
    {
        UE_LOG(LogTemp, Warning, TEXT("TestAppGameNetwork PostBox response for [%s]. Code: %d. Payload: %s"),
               *Request->GetURL(), Response->GetResponseCode(), *Response->GetContentAsString());
    }
    else
    {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4, TEXT( "TestAppGameNetwork PostBox response for [%s]. No response"),
                                                                         *Request->GetURL());
        if(responseListner->IsBound())
        {
            responseListner->Execute(false, rtnConnectResponse);
        }
        return;
    }
    
    TSharedPtr<FJsonObject> arg;
    TSharedRef< TJsonReader<> > reader = TJsonReaderFactory<>::Create( *Response->GetContentAsString() );
    
    if ( FJsonSerializer::Deserialize( reader, arg ) == false ){
        UE_LOG(LogTemp, Warning, TEXT("====>>>> parse failed : (%s)"), *Response->GetContentAsString() );
        if(responseListner->IsBound())
        {
            responseListner->Execute(false, rtnConnectResponse);
        }
        return;
    }

    
    FString resultCode = arg->GetStringField(TEXT("result_code"));
    
    TSharedPtr<FJsonObject> resultObjects;// = arg->GetObjectField(TEXT("result_data"));
    UE_LOG(LogTemp, Warning, TEXT("====>>>> resultObjects bf null check  "  ));
    if(arg->HasTypedField<EJson::Object>(TEXT("result_data")))
    {
        UE_LOG(LogTemp, Warning, TEXT("====>>>> resultObjects getObjectField OK  "  ));
        resultObjects = arg->GetObjectField(TEXT("result_data"));
    }
    else
    {
        UE_LOG(LogTemp, Warning, TEXT("====>>>> resultObjects getObjectField Failed..  "  ));
        resultObjects = MakeShareable(new FJsonObject);
    }
    
    // authKey
    FString authKey;
    if(resultObjects->TryGetStringField(TEXT("auth_key"), authKey)){
        UE_LOG(LogTemp, VeryVerbose, TEXT("is auth_key"));
        rtnConnectResponse.authKey = authKey;
    }
    else {
        UE_LOG(LogTemp, VeryVerbose, TEXT("is not auth_key"));
   }
    
    rtnConnectResponse.resultCode = resultCode;
    rtnConnectResponse.resultData = resultObjects;
    rtnConnectResponse.resultMsg = arg->GetStringField(TEXT("result_msg"));
    if(FCString::Atoi(*resultCode) == 200)
    {
        FString postBoxSessionToken = arg->GetStringField(TEXT("session_token"));
        rtnConnectResponse.sessionToken = postBoxSessionToken;
        if(responseListner->IsBound())
        {
            responseListner->Execute(true, rtnConnectResponse);
        }
    }
    else {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4, TEXT( "TestAppGameNetwork PostBox response for Result Code : %s Msg %s"), *resultCode, *(rtnConnectResponse.resultMsg));

        if(responseListner->IsBound())
        {
            responseListner->Execute(false, rtnConnectResponse);
        }
    }
}

void TestAppGameNetwork::verifyReceiptWithLogin(FGameListener listener, FString bypassReceipt)
{
    UTestAppUserGameInstance* UserGI = UTestAppUserGameInstance::GetInstance(AHIVESDKV4TesterGameMode::getGameModeInstance());
    if (UserGI->sessionToken.IsEmpty())
    {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4, TEXT("Session token is empty. Attempting to log in..."));

        // 로그인 호출 및 콜백 처리
        UserGI->loginWithCallback([listener, bypassReceipt, UserGI](bool loginResult, ITestAppGameNetworkResponse response) {
            if (loginResult && !UserGI->sessionToken.IsEmpty())
            {
                AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4, TEXT("Login successful. Proceeding with verifyReceipt."));
                TestAppGameNetwork::verifyReceipt(listener, bypassReceipt);
            }
            else
            {
                AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4, TEXT("Login failed or session token is still empty. Skipping verifyReceipt."));
                listener.ExecuteIfBound(false, response);
            }
            });
    }
    else
    {
        verifyReceipt(listener, bypassReceipt);
    }
}

