/* Copyright © 2025-Present Com2uS Platform Corp. All Rights Reserved. */
#include "HiveManagerFile.h"
#include "FileUtilities/ZipArchiveReader.h"
#include "Logger/HiveManagerLogger.h"

namespace {
	bool CreateDirectoryIfNotExists(const FString& DirectoryPath)
	{
		if (DirectoryPath.IsEmpty())
		{
			return false;
		}

		IPlatformFile& PlatformFile = FPlatformFileManager::Get().GetPlatformFile();
		
		if (!PlatformFile.DirectoryExists(*DirectoryPath))
		{
			return PlatformFile.CreateDirectoryTree(*DirectoryPath);
		}

		return true;
	}

	bool DeleteFileIfExists(const FString& FilePath)
	{
		if (FilePath.IsEmpty())
		{
			return false;
		}

		IPlatformFile& PlatformFile = FPlatformFileManager::Get().GetPlatformFile();

		if (PlatformFile.FileExists(*FilePath))
		{
			return PlatformFile.DeleteFile(*FilePath);
		}

		return true;
	}

	FString GetFileNameFromUrl(const FString& Url)
	{
		FString FileName = FPaths::GetCleanFilename(Url);

		int32 QueryIndex;
		if (FileName.FindChar(TEXT('?'), QueryIndex))
		{
			FileName = FileName.Left(QueryIndex);
		}

		return FileName;
	}
}

void HiveManagerFile::DownloadFile(
	const FString& DownloadUrl,
	const FString& DownloadTargetPath,
	FDownloadProgressDelegate ProgressCallback,
	FDownloadCompleteDelegate CompleteCallback
) {
	HM_LOG_WARNING(TEXT("Called DownloadFile() DownloadUrl= %s / DownloadTargetPath= %s"), *DownloadUrl, *DownloadTargetPath);

	TSharedRef<IHttpRequest, ESPMode::ThreadSafe> Request = FHttpModule::Get().CreateRequest();
	Request->SetURL(DownloadUrl);
	Request->SetVerb(TEXT("GET"));

	if (ProgressCallback.IsBound())
	{
		Request->OnRequestProgress64().BindLambda([ProgressCallback](FHttpRequestPtr Request, uint64 BytesSent, uint64 BytesReceived) {

			// SDK Manager Server 에서 Header 값에 Content-Length 를 내려주지 않아, 진행률 계산 불가
			// 단순 수신 데이터 사이즈만 콜백으로 전달
			ProgressCallback.Execute(BytesReceived);
		});
	}

	Request->OnProcessRequestComplete().BindLambda([DownloadTargetPath, CompleteCallback](FHttpRequestPtr HttpRequest, FHttpResponsePtr HttpResponse, bool bSucceeded) {

		if (!bSucceeded || !HttpResponse.IsValid())
		{
			HM_LOG_WARNING(TEXT("Failed DownloadFile(). HTTP Connection Error."));
			CompleteCallback.ExecuteIfBound(false, TEXT(""));
			return;
		}

		int32 StatusCode = HttpResponse->GetResponseCode();
		if (StatusCode < 200 || StatusCode >= 300)
		{
			HM_LOG_WARNING(TEXT("Failed DownloadFile(). HTTP Response Error: (Status: %d)"), StatusCode);
			CompleteCallback.ExecuteIfBound(false, TEXT(""));
			return;
		}

		FString Url = HttpRequest->GetURL();
		FString FileName = GetFileNameFromUrl(Url);
		FString DownloadFilePath = FPaths::Combine(DownloadTargetPath, FileName);

		HM_LOG_WARNING(TEXT("\n  FileName = %s\n  DownloadFilePath = %s"), *FileName, *DownloadFilePath);

		// create clean directory
		CreateDirectoryIfNotExists(DownloadTargetPath);
		DeleteFileIfExists(DownloadFilePath);

		// check available disk space
		uint64 TotalBytes = 0;
		uint64 FreeBytes = 0;
		if (!FPlatformMisc::GetDiskTotalAndFreeSpace(DownloadTargetPath, TotalBytes, FreeBytes))
		{
			HM_LOG_WARNING(TEXT("Get disk space error"));
			CompleteCallback.ExecuteIfBound(false, TEXT(""));
			return;
		}
		else
		{
			uint64 Bytes = FreeBytes;
			double MB = static_cast<double>(Bytes) / (1024.0 * 1024.0);

			HM_LOG(TEXT("Free space: %.2fMB"), MB);
		}

		
		// compare size
		const TArray<uint8>& Content = HttpResponse->GetContent();

		if (FreeBytes < static_cast<uint64>(Content.Num()))
		{
			HM_LOG_WARNING(TEXT("Not enough disk space."));
			CompleteCallback.ExecuteIfBound(false, TEXT(""));
			return;
		}
		
		
		// save file
		if (FFileHelper::SaveArrayToFile(Content, *DownloadFilePath))
		{
			HM_LOG_WARNING(TEXT("Save Succeeded"));
			CompleteCallback.ExecuteIfBound(true, DownloadFilePath);
		}
		else
		{
			HM_LOG_WARNING(TEXT("Save Failed"));
			CompleteCallback.ExecuteIfBound(false, TEXT(""));
		}
	});

	Request->ProcessRequest();
}

bool HiveManagerFile::UnZipp(const FString& Path, const FString& TargetPath)
{
	IPlatformFile& FileManager = FPlatformFileManager::Get().GetPlatformFile();

	IFileHandle* ArchiveFileHandle = FileManager.OpenRead(*Path);
	const FZipArchiveReader ZipArchiveReader(ArchiveFileHandle);
	if (!ZipArchiveReader.IsValid())
	{
		HM_LOG_WARNING(TEXT("Error opening archive file"));
		return false;
	}

	const TArray<FString> ArchiveFiles = ZipArchiveReader.GetFileNames();
	for (const FString& FileName : ArchiveFiles)
	{
		if (FileName.EndsWith("/") || FileName.EndsWith("\\"))
			continue;
		const FString AbsoluteDestFileName = TargetPath / FileName;
		if (!FPaths::IsUnderDirectory(AbsoluteDestFileName, TargetPath))
			continue;
		if (TArray<uint8> FileBuffer; ZipArchiveReader.TryReadFile(FileName, FileBuffer))
		{
			if (!FFileHelper::SaveArrayToFile(FileBuffer, *(TargetPath / FileName)))
			{
				HM_LOG_WARNING(TEXT("Error saving unarchived data to file"));
			}
		}
	}  

	return true;
}

FString HiveManagerFile::GetHashFromFile(const FString& FilePath)
{
	TArray<uint8> FileData;
	if (!FFileHelper::LoadFileToArray(FileData, *FilePath))
	{
		UE_LOG(LogTemp, Error, TEXT("Failed to load file: %s"), *FilePath);
		return FString();
	}

	FSHAHash Hash;
	FSHA1::HashBuffer(FileData.GetData(), FileData.Num(), Hash.Hash);

	return Hash.ToString().ToLower();
}

void HiveManagerFile::BackupPluginFiles(const TArray<FString>& PluginFiles, const FString& DirName, FBackupPluginFilesCompleteDelegate OnComplete)
{
	IPlatformFile& PlatformFile = FPlatformFileManager::Get().GetPlatformFile();
	TArray<FString> BackedUpFiles;

	FString PluginDir = FPaths::ProjectPluginsDir();
	FString TargetDir = FPaths::ProjectSavedDir() / TEXT("HiveSDKManager") / TEXT("Backup") / DirName;

	if (PlatformFile.DirectoryExists(*TargetDir))
	{
		if (!PlatformFile.DeleteDirectoryRecursively(*TargetDir))
		{
			HM_LOG_WARNING(TEXT("Failed to delete back-up dir : %s"), *TargetDir);
			OnComplete.ExecuteIfBound(false, BackedUpFiles);
			return;
		}
	}

	if (!PlatformFile.CreateDirectoryTree(*TargetDir))
	{
		HM_LOG_WARNING(TEXT("Failed to create back-up dir : %s"), *TargetDir);
		OnComplete.ExecuteIfBound(false, BackedUpFiles);
		return;
	}

	for (const FString& File : PluginFiles)
	{
		FString SrcFilePath = PluginDir / File;
		FString DestFilePath = TargetDir / File;

		FString DestDir = FPaths::GetPath(DestFilePath); // TargetDir는 루트 경로이며, DestDir는 해당 루트 아래 파일이 위치할 실제 목적지 디렉토리 경로

		if (!PlatformFile.DirectoryExists(*DestDir))
		{
			if (!PlatformFile.CreateDirectoryTree(*DestDir))
			{
				HM_LOG_WARNING(TEXT("Failed to create dest dir : %s"), *DestDir);
				OnComplete.ExecuteIfBound(false, BackedUpFiles);
				return;
			}
		}

		// 원본 파일 체크
		if (!PlatformFile.FileExists(*SrcFilePath)) {
			// 특정 파일을 직접 지운 경우, 실제 파일이 존재하지 않을 수 있음. 에러 상황은 아님.
			HM_LOG(TEXT("File is not exist. %s"), *SrcFilePath);
			continue;
		}

		// 원본 파일 복사
		if (!PlatformFile.CopyFile(*DestFilePath, *SrcFilePath))
		{
			HM_LOG_WARNING(TEXT("Failed copy file to %s from %s"), *DestFilePath, *SrcFilePath);
			OnComplete.ExecuteIfBound(false, BackedUpFiles);
			return;
		}

		BackedUpFiles.Add(File);
	}
	OnComplete.ExecuteIfBound(true, BackedUpFiles);
}

void HiveManagerFile::RollbackPluginFiles(const TArray<FString>& PluginFiles, const FString& DirName)
{
	IPlatformFile& PlatformFile = FPlatformFileManager::Get().GetPlatformFile();

	FString PluginDir = FPaths::ProjectPluginsDir();
	FString BackupDir = FPaths::ProjectSavedDir() / TEXT("HiveSDKManager") / TEXT("Backup") / DirName;

	for (const FString& File : PluginFiles)
	{
		FString SrcFilePath = BackupDir / File;
		FString DestFilePath = PluginDir / File;

		if (!PlatformFile.FileExists(*SrcFilePath))
		{
			HM_LOG_WARNING(TEXT("SrcFile is not exist. %s"), *SrcFilePath);
			continue;
		}

		if (PlatformFile.FileExists(*DestFilePath))
		{
			if (!PlatformFile.DeleteFile(*DestFilePath))
			{
				HM_LOG_WARNING(TEXT("Failed to delete dest file %s"), *DestFilePath);
				continue;
			}
		}

		PlatformFile.MoveFile(*DestFilePath, *SrcFilePath);
	}
}