﻿/* Copyright © 2025-Present Com2uS Platform Corp. All Rights Reserved. */

#include "HiveSDKManagerSubsystem.h"
#include "HiveManagerProtocolApi.h"
#include "HiveManagerFile.h"
#include "HiveSDKManagerConfig.h"
#include "HiveManagerIgnorePathFileManager.h"
#include "Logger/HiveManagerLogger.h"

#include "Json.h"
#include "JsonUtilities.h"

namespace {
    
    // if(Left > Right) return positive.
    // if(Left < Right) return negative.
    // if(Left ==Right) return 0.
    int32 CompareVersionStrings(const FString& Left, const FString& Right)
    {
        TArray<FString> LeftParts;
        TArray<FString> RightParts;

        Left.ParseIntoArray(LeftParts, TEXT("."));
        Right.ParseIntoArray(RightParts, TEXT("."));

        int32 MaxLength = FMath::Max(LeftParts.Num(), RightParts.Num());

        for (int32 i = 0; i < MaxLength; ++i)
        {
            int32 LeftNum = (i < LeftParts.Num()) ? FCString::Atoi(*LeftParts[i]) : 0;
            int32 RightNum = (i < RightParts.Num()) ? FCString::Atoi(*RightParts[i]) : 0;

            if (LeftNum != RightNum)
            {
                return LeftNum - RightNum;
            }
        }

        return 0;
    }

    bool CheckFileValidation(const FString& FilePath, const FString& ValidHash)
    {
        IPlatformFile& PlatformFile = FPlatformFileManager::Get().GetPlatformFile();

        if (!PlatformFile.FileExists(*FilePath))
        {
            return false;
        }

        return ValidHash.Equals(HiveManagerFile::GetHashFromFile(FilePath), ESearchCase::IgnoreCase);
    }
}

void UHiveSDKManagerSubsystem::Initialize(FSubsystemCollectionBase& Collection)
{
    HM_LOG(TEXT("Called Initialize"));
    UpdateAvailableInterface();
    UpdateAvailablePlatform();

    HiveManagerStrings = NewObject<UHiveManagerStrings>(this);
    HiveManagerStrings->Initialize();
}

void UHiveSDKManagerSubsystem::Deinitialize()
{
    HM_LOG(TEXT("Called Deinitialize"));
}

FString UHiveSDKManagerSubsystem::GetCurrentVersion(const FString& Type)
{
    // Validate input type
    FString LowerType = Type.ToLower();
    if (LowerType != TEXT("interface") &&
        LowerType != TEXT("android") &&
        LowerType != TEXT("ios") &&
        LowerType != TEXT("windows"))
    {
        HM_LOG_WARNING(TEXT("GetCurrentVersion: Invalid parameter type '%s'."), *Type);
        return FString();
    }

    // Build path to version.json based on type
    FString PluginDir = FPaths::ProjectPluginsDir() / TEXT("HIVESDK");
    FString SourcePath;

    if (LowerType == TEXT("interface"))
    {
        SourcePath = PluginDir / TEXT("Source/HIVESDK");
    }
    else if (LowerType == TEXT("android"))
    {
        SourcePath = PluginDir / TEXT("Source/HiveSDKAndroid");
    }
    else if (LowerType == TEXT("ios"))
    {
        SourcePath = PluginDir / TEXT("Source/HiveSDKiOS");
    }
    else if (LowerType == TEXT("windows"))
    {
        SourcePath = PluginDir / TEXT("Source/HiveSDKWindows");
    }

    FString VersionFilePath = SourcePath / TEXT("version.json");

    // Check if file exists
    if (!FPaths::FileExists(VersionFilePath))
    {
        HM_LOG_WARNING(TEXT("GetCurrentVersion: version.json not found at '%s'"), *VersionFilePath);
        return FString();
    }

    // Read and parse version.json
    FString Version = ReadVersionFromJson(VersionFilePath);

    if (Version.IsEmpty())
    {
        HM_LOG_WARNING(TEXT("GetCurrentVersion: Failed to read version from '%s'"), *VersionFilePath);
    }

    return Version;
}

bool UHiveSDKManagerSubsystem::IsHiveSDKInstalled()
{
    FString CurrentVersion = GetCurrentVersion(TEXT("interface"));
    return !CurrentVersion.IsEmpty();
}

bool UHiveSDKManagerSubsystem::IsInitialized()
{
    if (CachedAvailableInterface.Num() == 0)
    {
        return false;
    }
    else {
        return true;
    }
}

FString UHiveSDKManagerSubsystem::GetAllCurrentVersionForUI()
{
    FString InterfaceVersion = GetCurrentVersion(TEXT("interface"));
    FString AndroidVersion = GetCurrentVersion(TEXT("android"));
    FString iOSVersion = GetCurrentVersion(TEXT("ios"));
    FString WindowsVersion = GetCurrentVersion(TEXT("windows"));
    FString MacVersion = GetCurrentVersion(TEXT("mac"));

    if (InterfaceVersion.IsEmpty()) InterfaceVersion = TEXT("-");
    if (AndroidVersion.IsEmpty()) AndroidVersion = TEXT("-");
    if (iOSVersion.IsEmpty()) iOSVersion = TEXT("-");
    if (WindowsVersion.IsEmpty()) WindowsVersion = TEXT("-");
    if (MacVersion.IsEmpty()) MacVersion = TEXT("-");

    FString Result = FString::Printf(
        TEXT("Interface : %s\n")
        TEXT("Android : %s\n")
        TEXT("iOS : %s\n")
        TEXT("Windows : %s\n")
        TEXT("Mac : %s"),
        *InterfaceVersion,
        *AndroidVersion,
        *iOSVersion,
        *WindowsVersion,
        *MacVersion
    );

    return Result;
}

FString UHiveSDKManagerSubsystem::GetAllLatestVersionForUI()
{
    FString InterfaceVersion = GetLatestAvailableInterfaceVersion();
    FString AndroidVersion = GetLatestAvailablePlatformVersion(TEXT("android"));
    FString iOSVersion = GetLatestAvailablePlatformVersion(TEXT("ios"));
    FString WindowsVersion = GetLatestAvailablePlatformVersion(TEXT("windows"));
    FString MacVersion = GetLatestAvailablePlatformVersion(TEXT("mac"));

    if (InterfaceVersion.IsEmpty()) InterfaceVersion = TEXT("-");
    if (AndroidVersion.IsEmpty()) AndroidVersion = TEXT("-");
    if (iOSVersion.IsEmpty()) iOSVersion = TEXT("-");
    if (WindowsVersion.IsEmpty()) WindowsVersion = TEXT("-");
    if (MacVersion.IsEmpty()) MacVersion = TEXT("-");

    FString Result = FString::Printf(
        TEXT("Interface : %s\n")
        TEXT("Android : %s\n")
        TEXT("iOS : %s\n")
        TEXT("Windows : %s\n")
        TEXT("Mac : %s"),
        *InterfaceVersion,
        *AndroidVersion,
        *iOSVersion,
        *WindowsVersion,
        *MacVersion
    );

    return Result;
}

FString UHiveSDKManagerSubsystem::GetManagerVersion()
{
    return FString(HiveConfig::Version);
}

void UHiveSDKManagerSubsystem::GetNotice()
{
	TWeakObjectPtr<UHiveSDKManagerSubsystem> WeakThis(this);
	FHiveManagerProtocolApi::RequestNoticeInfo([WeakThis](bool bSucceeded, TArray<FManagerNoticeInfo> ResponseData, const FString& ErrorMessage)
		{
			if (!WeakThis.IsValid())
			{
				return;
			}

			if (bSucceeded)
			{
				ResponseData.Sort([](const FManagerNoticeInfo& A, const FManagerNoticeInfo& B)
					{
						return A.StartDate > B.StartDate;
					});

				TArray<UManagerNoticeInfoViewModel*> Top6Notices;
				int32 Count = FMath::Min(6, ResponseData.Num());
				for (int32 i = 0; i < Count; ++i)
				{
					Top6Notices.Add(UManagerNoticeInfoViewModel::FromDomainModel(WeakThis.Get(), ResponseData[i]));
				}

				WeakThis->OnNoticeInfoReceived.Broadcast(Top6Notices);
			}
			else
			{
                HM_LOG_WARNING(TEXT("Failed to get notice info: %s"), *ErrorMessage);
			}
		});
}

FString UHiveSDKManagerSubsystem::ReadVersionFromJson(const FString& FilePath)
{
    // Load file content
    FString JsonString;
    if (!FFileHelper::LoadFileToString(JsonString, *FilePath))
    {
        HM_LOG_WARNING(TEXT("ReadVersionFromJson: Failed to load file '%s'"), *FilePath);
        return FString();
    }

    // Parse JSON
    TSharedPtr<FJsonObject> JsonObject;
    TSharedRef<TJsonReader<>> Reader = TJsonReaderFactory<>::Create(JsonString);

    if (!FJsonSerializer::Deserialize(Reader, JsonObject) || !JsonObject.IsValid())
    {
        HM_LOG_WARNING(TEXT("ReadVersionFromJson: Failed to parse JSON from '%s'"), *FilePath);
        return FString();
    }

    // Extract version field
    FString Version;
    if (!JsonObject->TryGetStringField(TEXT("version"), Version))
    {
        HM_LOG_WARNING(TEXT("ReadVersionFromJson: 'version' field not found in '%s'"), *FilePath);
        return FString();
    }

    return Version;
}

void UHiveSDKManagerSubsystem::UpdateAvailableInterface()
{
    TWeakObjectPtr<UHiveSDKManagerSubsystem> WeakThis(this);

	FHiveManagerProtocolApi::RequestInterfaceAll([WeakThis](bool bSucceeded, TArray<FManagerInterfaceInfo> ResponseData, const FString& ErrorMessage)
		{
            if (!WeakThis.IsValid())
            {
                HM_LOG_WARNING(TEXT("UpdateAvailableInterface: Failed WeakThis is invalid."));
                return;
            }

            if (!bSucceeded)
            {
                HM_LOG_WARNING(TEXT("UpdateAvailableInterface: Failed (%s)"), *ErrorMessage);
                WeakThis->OnUpdateAvailableInterfaceCompleted.ExecuteIfBound(false);
                return;
            }

            TArray<FManagerInterfaceInfo> FilteredArray = ResponseData.FilterByPredicate([](const FManagerInterfaceInfo& Info)
                {
                    return Info.State == 1;
                }
            );

            if (WeakThis.IsValid())
            {
                WeakThis->CachedAvailableInterface = MoveTemp(FilteredArray);
                WeakThis->OnUpdateAvailableInterfaceCompleted.ExecuteIfBound(true);
            }
		});
}

void UHiveSDKManagerSubsystem::UpdateAvailablePlatform()
{
    TArray<FString> Platforms = { TEXT("android"), TEXT("ios"), TEXT("windows"), TEXT("mac") };
    TWeakObjectPtr<UHiveSDKManagerSubsystem> WeakThis(this);

    UpdateAvailablePlatform_Recursive(Platforms, 0, WeakThis, true);
}

void UHiveSDKManagerSubsystem::UpdateAvailablePlatform_Recursive(const TArray<FString>& Platforms, int32 Index, TWeakObjectPtr<UHiveSDKManagerSubsystem> WeakThis, bool bSoFarSucceeded)
{
    if (!WeakThis.IsValid())
    {
        HM_LOG_WARNING(TEXT("UpdateAvailablePlatform_Recursive: Failed WeakThis is invalid."));
        return;
    }

    if (Index >= Platforms.Num())
    {
        WeakThis->OnUpdateAvailablePlatformCompleted.ExecuteIfBound(bSoFarSucceeded);
        return;
    }

    const FString& Platform = Platforms[Index];
    FHiveManagerProtocolApi::RequestPlatformAll(
        *Platform,
        [WeakThis, Platforms, Index, bSoFarSucceeded](bool bSucceeded, TArray<FManagerPlatformInfo> ResponseData, const FString& ErrorMessage) mutable
        {
            if (!WeakThis.IsValid())
            {
                HM_LOG_WARNING(TEXT("UpdateAvailablePlatform_Recursive: Failed WeakThis is invalid(2)."));
                return;
            }

            bool bCurrentSucceeded = bSoFarSucceeded && bSucceeded;

            if (!bSucceeded)
            {
                HM_LOG_WARNING(TEXT("UpdateAvailablePlatform: Failed RequestPlatformAll(%s) - %s"), *Platforms[Index], *ErrorMessage);
                WeakThis->OnUpdateAvailablePlatformCompleted.ExecuteIfBound(bSoFarSucceeded);
                return;
            }
            
			TArray<FManagerPlatformInfo> FilteredArray = ResponseData.FilterByPredicate([](const FManagerPlatformInfo& Info)
				{
					return Info.State == 1;
				}
			);

			if (Platforms[Index] == TEXT("android"))
			{
				WeakThis->CachedAvailableAndroid = MoveTemp(FilteredArray);
			}
			else if (Platforms[Index] == TEXT("ios"))
			{
				WeakThis->CachedAvailableiOS = MoveTemp(FilteredArray);
			}
			else if (Platforms[Index] == TEXT("windows"))
			{
				WeakThis->CachedAvailableWindows = MoveTemp(FilteredArray);
			}
			else if (Platforms[Index] == TEXT("mac"))
			{
				WeakThis->CachedAvailableMac = MoveTemp(FilteredArray);
			}

            if (WeakThis.IsValid())
            {
                WeakThis->UpdateAvailablePlatform_Recursive(Platforms, Index + 1, WeakThis, bCurrentSucceeded);
            }
        }
    );
}

FManagerInterfaceInfo* UHiveSDKManagerSubsystem::GetLatestManagerInterfaceInfo()
{
    if (CachedAvailableInterface.Num() == 0)
    {
        return nullptr;
    }

    FManagerInterfaceInfo* LatestInfo = &CachedAvailableInterface[0];

    for (int32 i = 1; i < CachedAvailableInterface.Num(); ++i)
    {
        if (CompareVersionStrings(CachedAvailableInterface[i].Version, LatestInfo->Version) > 0)
        {
            LatestInfo = &CachedAvailableInterface[i];
        }
    }

    return LatestInfo;
}

FString UHiveSDKManagerSubsystem::GetLatestAvailableInterfaceVersion()
{
    FManagerInterfaceInfo* LatestInfo = GetLatestManagerInterfaceInfo();
    return LatestInfo ? LatestInfo->Version : FString();
}

FManagerPlatformInfo* UHiveSDKManagerSubsystem::GetLatestManagerPlatformInfo(const FString& Platform)
{
    TArray<FManagerPlatformInfo>* TargetArray = nullptr;

    if (Platform.Equals(TEXT("android"), ESearchCase::IgnoreCase))
    {
        TargetArray = &CachedAvailableAndroid;
    }
    else if (Platform.Equals(TEXT("ios"), ESearchCase::IgnoreCase))
    {
        TargetArray = &CachedAvailableiOS;
    }
    else if (Platform.Equals(TEXT("windows"), ESearchCase::IgnoreCase))
    {
        TargetArray = &CachedAvailableWindows;
    }
    else if (Platform.Equals(TEXT("mac"), ESearchCase::IgnoreCase))
    {
        TargetArray = &CachedAvailableMac;
    }
    else
    {
        return nullptr;
    }

    if (TargetArray->Num() == 0)
    {
        return nullptr;
    }

    FManagerPlatformInfo* LatestInfo = &(*TargetArray)[0];

    for (int32 i = 1; i < TargetArray->Num(); ++i)
    {
        if (CompareVersionStrings((*TargetArray)[i].Version, LatestInfo->Version) > 0)
        {
            LatestInfo = &(*TargetArray)[i];
        }
    }

    return LatestInfo;
}

FString UHiveSDKManagerSubsystem::GetLatestAvailablePlatformVersion(const FString& Platform)
{
    FManagerPlatformInfo* LatestInfo = GetLatestManagerPlatformInfo(Platform);
    return LatestInfo ? LatestInfo->Version : FString();
}

void UHiveSDKManagerSubsystem::IntegrityVerificationInterface(FOnIntegrityVerificationInterfaceComplete OnComplete)
{
    HM_LOG(TEXT("Called IntegrityVerificationInterface"));

    TWeakObjectPtr<UHiveSDKManagerSubsystem> WeakThis(this);
	FHiveManagerProtocolApi::RequestInterfaceDetail(
		GetCurrentVersion(TEXT("interface")),
		[OnComplete, WeakThis](bool bSucceeded, FManagerInterfaceDetailInfo ResponseData, const FString& ErrorMessage)
        {
            TArray<FString> InvalidFiles;

			if (!bSucceeded) {
                HM_LOG_WARNING(TEXT("Failed IntegrityVerificationInterface"));
                OnComplete.ExecuteIfBound(InvalidFiles, ResponseData);
				return;
			}
            
            for (const FFile& File : ResponseData.Files)
            {
                FString CurrentFilePath = FPaths::Combine(FPaths::ProjectPluginsDir(), File.Path);

                if (!CheckFileValidation(CurrentFilePath, File.FileHash))
                {
                    HM_LOG_WARNING(TEXT("IntegrityVerificationInterface diff file : %s"), *CurrentFilePath);
                    InvalidFiles.Add(File.Path);
                }
            }

            TArray<FString> NonIgnoredInvalidFiles;

            if (WeakThis.IsValid())
            {
                NonIgnoredInvalidFiles = WeakThis->FilterIgnoredFiles(InvalidFiles);
            }

            OnComplete.ExecuteIfBound(NonIgnoredInvalidFiles, ResponseData);
		}
	);
}

void UHiveSDKManagerSubsystem::IntegrityVerificationPlatform(const FString& Platform, FOnIntegrityVerificationPlatformComplete OnComplete)
{
    TWeakObjectPtr<UHiveSDKManagerSubsystem> WeakThis(this);
    FHiveManagerProtocolApi::RequestPlatformDetail(
        Platform,
        GetCurrentVersion(Platform),
        [OnComplete, WeakThis](bool bSucceeded, FManagerPlatformDetailInfo ResponseData, const FString& ErrorMessage)
        {
            TArray<FString> InvalidFiles;

            if (!bSucceeded) {
                HM_LOG_WARNING(TEXT("Failed IntegrityVerificationPlatform"));
                OnComplete.ExecuteIfBound(InvalidFiles, ResponseData);
                return;
            }

            for (const FFile& File : ResponseData.Files)
            {
                FString CurrentFilePath = FPaths::Combine(FPaths::ProjectPluginsDir(), File.Path);

                if (!CheckFileValidation(CurrentFilePath, File.FileHash))
                {
                    HM_LOG_WARNING(TEXT("IntegrityVerificationPlatform diff file : %s"), *CurrentFilePath);
                    InvalidFiles.Add(File.Path);
                }
            }

            TArray<FString> NonIgnoredInvalidFiles;

            if (WeakThis.IsValid())
            {
                NonIgnoredInvalidFiles = WeakThis->FilterIgnoredFiles(InvalidFiles);
            }

            OnComplete.ExecuteIfBound(NonIgnoredInvalidFiles, ResponseData);
        }
    );
}

void UHiveSDKManagerSubsystem::IntegrityVerification(FOnIntegrityVerificationComplete OnComplete)
{
    struct FIntegrityVerificationState
    {
        TArray<FString> CombinedInvalidFiles;
        int32 CompletedCount = 0;
        int32 TotalCount = 5; // Interface + 4 Platforms[android, ios, windows, mac]
        FCriticalSection Mutex;
    };

    TSharedPtr<FIntegrityVerificationState> SharedState = MakeShared<FIntegrityVerificationState>();

    // Check if all verifications are complete and execute callback
    auto CheckAndExecuteCallback = [SharedState, OnComplete]()
        {
            FScopeLock Lock(&SharedState->Mutex);
            SharedState->CompletedCount++;

            if (SharedState->CompletedCount >= SharedState->TotalCount)
            {
                OnComplete.ExecuteIfBound(SharedState->CombinedInvalidFiles);
            }
        };

    IntegrityVerificationInterface(
        FOnIntegrityVerificationInterfaceComplete::CreateLambda(
            [SharedState, CheckAndExecuteCallback](const TArray<FString>& InvalidFiles, const FManagerInterfaceDetailInfo& ManagerInterfaceDetailInfo)
            {
                FScopeLock Lock(&SharedState->Mutex);
                SharedState->CombinedInvalidFiles.Append(InvalidFiles);
                CheckAndExecuteCallback();
            }
        )
    );

    
    TArray<FString> Platforms = { TEXT("android"), TEXT("ios"), TEXT("windows"), TEXT("mac") };

    for (const FString& Platform : Platforms)
    {
        IntegrityVerificationPlatform(
            Platform,
            FOnIntegrityVerificationPlatformComplete::CreateLambda(
                [SharedState, CheckAndExecuteCallback](const TArray<FString>& InvalidFiles, const FManagerPlatformDetailInfo& ManagerPlatformDetailInfo)
                {
                    FScopeLock Lock(&SharedState->Mutex);
                    SharedState->CombinedInvalidFiles.Append(InvalidFiles);
                    CheckAndExecuteCallback();
                }
            )
        );
    }
}

void UHiveSDKManagerSubsystem::IntegrityVerificationBP()
{
    TWeakObjectPtr<UHiveSDKManagerSubsystem> WeakThis(this);
	IntegrityVerification(
		FOnIntegrityVerificationComplete::CreateLambda(
			[WeakThis](const TArray<FString>& InvalidFiles) {
                WeakThis->OnIntegrityVerificationCompleteDelegate.Broadcast(InvalidFiles);
			}
		)
	);
}

void UHiveSDKManagerSubsystem::RetryToConnectManagerServer()
{
    TWeakObjectPtr<UHiveSDKManagerSubsystem> WeakThis(this);
	FHiveManagerProtocolApi::RequestManagerServerState(
		[WeakThis](bool bSucceeded, FManagerEmptyResponse ResponseData, const FString& ErrorMessage)
		{
            if (!bSucceeded) {
                HM_LOG_WARNING(TEXT("Failed RetryToConnectManagerServer"));
                if (WeakThis.IsValid())
                {
                    WeakThis->OnRetryToConnectManagerServerDelegate.Broadcast(false);
                }
                return;
            }

            if (WeakThis.IsValid())
            {
                WeakThis->UpdateAvailableInterface();
                WeakThis->UpdateAvailablePlatform();
                WeakThis->OnRetryToConnectManagerServerDelegate.Broadcast(true);
            }
		}
	);
}

void UHiveSDKManagerSubsystem::EditIgnorePatchFile()
{
    FHiveManagerIgnorePatchFileManager::Get().EditIgnoreFile();
}

void UHiveSDKManagerSubsystem::Restore()
{
    struct FRestoreState
    {
        int32 CompletedCount = 0;
        int32 TotalCount = 5; // Interface + 4 Platforms[android, ios, windows, mac]
        FCriticalSection Mutex;
    };

    TSharedPtr<FRestoreState> SharedState = MakeShared<FRestoreState>();
    TWeakObjectPtr<UHiveSDKManagerSubsystem> WeakThis(this);

    auto CheckAndExecuteDelegate = [SharedState, WeakThis]()
        {
            FScopeLock Lock(&SharedState->Mutex);
            SharedState->CompletedCount++;

            if (SharedState->CompletedCount >= SharedState->TotalCount)
            {
                if (WeakThis.IsValid())
                {
                    WeakThis->OnRestoreComplete.Broadcast();
                }
            }
        };

    ResetoreInterface(FOnRestoreComplete::CreateLambda([CheckAndExecuteDelegate](bool bSucceeded, const FString& ErrorMessage)
        {
            CheckAndExecuteDelegate();
        }
    ));
    
    
    TArray<FString> Platforms = { TEXT("android"), TEXT("ios"), TEXT("windows"), TEXT("mac") };

    for (const FString& Platform : Platforms)
    {
        RestorePlatform(Platform, FOnRestoreComplete::CreateLambda([CheckAndExecuteDelegate](bool bSucceeded, const FString& ErrorMessage)
            {
                CheckAndExecuteDelegate();
            }
        ));
    }
}

void UHiveSDKManagerSubsystem::ResetoreInterface(FOnRestoreComplete OnComplete)
{
    TWeakObjectPtr<UHiveSDKManagerSubsystem> WeakThis(this);
    IntegrityVerificationInterface(FOnIntegrityVerificationInterfaceComplete::CreateLambda([WeakThis, OnComplete](const TArray<FString>& InvalidFiles, const FManagerInterfaceDetailInfo& ManagerInterfaceDetailInfo)
        {
            if (InvalidFiles.IsEmpty())
            {
                HM_LOG_WARNING(TEXT("NonIgnoredFiles is empty()"));
                OnComplete.ExecuteIfBound(true, TEXT("No files to restore for interface"));
                return;
            }

            if (WeakThis.IsValid())
            {
                WeakThis->ApplyPatch(
                    TEXT("Interface"),
                    InvalidFiles,
                    ManagerInterfaceDetailInfo.DownloadUrl,
                    FOnApplyPatchComplete::CreateLambda([OnComplete](bool bSucceeded, const FString& ErrorMessage)
                        {
                            OnComplete.ExecuteIfBound(bSucceeded, ErrorMessage);
                        }
                    ));
            }
        }
    ));
}

void UHiveSDKManagerSubsystem::RestorePlatform(const FString& Platform, FOnRestoreComplete OnComplete)
{
    TWeakObjectPtr<UHiveSDKManagerSubsystem> WeakThis(this);
    IntegrityVerificationPlatform(Platform, FOnIntegrityVerificationPlatformComplete::CreateLambda([WeakThis, Platform, OnComplete](const TArray<FString>& InvalidFiles, const FManagerPlatformDetailInfo& ManagerPlatformDetailInfo)
        {
            if (InvalidFiles.IsEmpty())
            {
                HM_LOG_WARNING(TEXT("NonIgnoredFiles is empty()"));
                OnComplete.ExecuteIfBound(true, TEXT("No files to restore for platform"));
                return;
            }

            if (WeakThis.IsValid())
            {
                WeakThis->ApplyPatch(
                    Platform,
                    InvalidFiles,
                    ManagerPlatformDetailInfo.DownloadUrl,
                    FOnApplyPatchComplete::CreateLambda([OnComplete](bool bSucceeded, const FString& ErrorMessage)
                        {
                            OnComplete.ExecuteIfBound(bSucceeded, ErrorMessage);
                        }
                    ));
            }
        }));
    
}

TArray<FString> UHiveSDKManagerSubsystem::FilterIgnoredFiles(const TArray<FString>& Files)
{
    TArray<FString> NonIgnoredFiles;

    FHiveManagerIgnorePatchFileManager::Get().LoadIgnore();

    for (const FString& File : Files)
    {
        if (!FHiveManagerIgnorePatchFileManager::Get().IsIgnore(File))
        {
            NonIgnoredFiles.Add(File);
        }
    }

    return NonIgnoredFiles;
}

void UHiveSDKManagerSubsystem::ApplyPatch(const FString& Type, const TArray<FString>& TargetFiles, const FString& DownloadUrl, FOnApplyPatchComplete OnComplete)
{
    // download
	FString DownloadPath = FPaths::ProjectSavedDir() / TEXT("HiveSDKManager") / TEXT("Download");

	IPlatformFile& PlatformFile = FPlatformFileManager::Get().GetPlatformFile();
	if (PlatformFile.DirectoryExists(*DownloadPath))
	{
		PlatformFile.DeleteDirectoryRecursively(*DownloadPath);
	}

    FScopedSlowTask PreprocessingSlowTask(3.f, FText::FromString("Preprocessing: Downloading, Unzipping, and Backing Up..."));
    PreprocessingSlowTask.MakeDialog();
    FPlatformProcess::Sleep(1.f);

	HiveManagerFile::DownloadFile(
		DownloadUrl,
		DownloadPath,
		FDownloadProgressDelegate::CreateLambda([](uint64 BytesReceived) {}),
		FDownloadCompleteDelegate::CreateLambda([Type, TargetFiles, OnComplete](bool bSucceeded, const FString& DownloadFilePath)
			{
                if (!bSucceeded)
				{
                    OnComplete.ExecuteIfBound(false, TEXT("Failed to download file"));
                    HM_LOG_WARNING(TEXT("Failed to download file"));
					return;
				}

				// unzip
				FString UnzipPath = FPaths::ProjectSavedDir() / TEXT("HiveSDKManager") / TEXT("Unzip");

				IPlatformFile& PlatformFile = FPlatformFileManager::Get().GetPlatformFile();
				if (PlatformFile.DirectoryExists(*UnzipPath))
				{
					PlatformFile.DeleteDirectoryRecursively(*UnzipPath);
				}

				if (!HiveManagerFile::UnZipp(DownloadFilePath, UnzipPath))
				{
                    OnComplete.ExecuteIfBound(false, TEXT("Failed to unzip"));
                    HM_LOG_WARNING(TEXT("Failed to unzip"));
					return;
				}

				// back-up

                FString BackupDirName = Type;
				HiveManagerFile::BackupPluginFiles(
					TargetFiles,
                    BackupDirName,
					FBackupPluginFilesCompleteDelegate::CreateLambda([TargetFiles, UnzipPath, BackupDirName, OnComplete](bool bAllSucceeded, const TArray<FString>& BackedUpFiles)
						{
							if (!bAllSucceeded)
							{
                                OnComplete.ExecuteIfBound(false, TEXT("Failed to back-up"));
                                HM_LOG_WARNING(TEXT("Failed to back-up"));
								return;
							}

							// patch
							IPlatformFile& PlatformFile = FPlatformFileManager::Get().GetPlatformFile();

                            FScopedSlowTask PatchSlowTask(TargetFiles.Num(), FText::FromString("Patch files..."));
                            PatchSlowTask.MakeDialog();
                            
                            FPlatformProcess::Sleep(1.f);

							for (const FString& File : TargetFiles) // Added, removed, modified items
							{
                                PatchSlowTask.EnterProgressFrame();

								FString SrcFilePath = UnzipPath / File;
								FString DestFilePath = FPaths::ProjectPluginsDir() / File;


                                // patch - 1. create destination dir
                                FString DestDir = FPaths::GetPath(DestFilePath);
                                if (!PlatformFile.DirectoryExists(*DestDir))
                                {
                                    if (!PlatformFile.CreateDirectoryTree(*DestDir))
                                    {
                                        HiveManagerFile::RollbackPluginFiles(TargetFiles, BackupDirName);
                                        HM_LOG_WARNING(TEXT("Failed to patch"));
                                        OnComplete.ExecuteIfBound(false, TEXT("Failed to patch"));
                                        return;
                                    }
                                }

                                // patch - 2. verify source file exist
                                if (!PlatformFile.FileExists(*SrcFilePath))
                                {
                                    HiveManagerFile::RollbackPluginFiles(TargetFiles, BackupDirName);
                                    HM_LOG_WARNING(TEXT("Failed to patch"));
                                    OnComplete.ExecuteIfBound(false, TEXT("Failed to patch"));
                                    return;
                                }

                                // patch - 3. delete existing destination file
                                if (PlatformFile.FileExists(*DestFilePath))
                                {
                                    if (!PlatformFile.DeleteFile(*DestFilePath))
                                    {
                                        HiveManagerFile::RollbackPluginFiles(TargetFiles, BackupDirName);
                                        HM_LOG_WARNING(TEXT("Failed to patch"));
                                        OnComplete.ExecuteIfBound(false, TEXT("Failed to patch"));
                                        return;
                                    }
                                }

                                // patch - 4. move file
                                if (!PlatformFile.CopyFile(*DestFilePath, *SrcFilePath))
                                {
                                    HiveManagerFile::RollbackPluginFiles(TargetFiles, BackupDirName);
                                    HM_LOG_WARNING(TEXT("Failed to patch"));
                                    OnComplete.ExecuteIfBound(false, TEXT("Failed to patch"));
                                    return;
                                }
							}

                            OnComplete.ExecuteIfBound(true, TEXT("Success"));
						}
					));
			})
	);
}

UUpdateListViewModel* UHiveSDKManagerSubsystem::GetUpdateListViewData()
{
    return UpdateListViewRawData;
}

void UHiveSDKManagerSubsystem::UpdateUpdateListViewData()
{
    UpdateListViewRawData = nullptr;
    UpdateListViewRawData = NewObject<UUpdateListViewModel>(this);

    // set current version
    TMap<FString, FString> CurrentVersionMap;
    CurrentVersionMap.Add(TEXT("Interface"), GetCurrentVersion(TEXT("Interface")));
    CurrentVersionMap.Add(TEXT("Android"), GetCurrentVersion(TEXT("Android")));
    CurrentVersionMap.Add(TEXT("iOS"), GetCurrentVersion(TEXT("iOS")));
    CurrentVersionMap.Add(TEXT("Windows"), GetCurrentVersion(TEXT("Windows")));
    CurrentVersionMap.Add(TEXT("Mac"), GetCurrentVersion(TEXT("Mac")));

    // set latest version
    TMap<FString, FString> LatestVersionMap;
    LatestVersionMap.Add(TEXT("Interface"), GetLatestAvailableInterfaceVersion());
    LatestVersionMap.Add(TEXT("Android"), GetLatestAvailablePlatformVersion(TEXT("Android")));
    LatestVersionMap.Add(TEXT("iOS"), GetLatestAvailablePlatformVersion(TEXT("iOS")));
    LatestVersionMap.Add(TEXT("Windows"), GetLatestAvailablePlatformVersion(TEXT("Windows")));
    LatestVersionMap.Add(TEXT("Mac"), GetLatestAvailablePlatformVersion(TEXT("Mac")));

    FString CurrentInterfaceVersion = CurrentVersionMap[TEXT("Interface")];

    TArray<UUpdateListViewItem*> Items;

    TArray<FManagerInterfaceInfo> SortedInterfaces = CachedAvailableInterface;
    SortedInterfaces.Sort([this](const FManagerInterfaceInfo& A, const FManagerInterfaceInfo& B) {
        return CompareVersionStrings(A.Version, B.Version) > 0;
        });

    for (const FManagerInterfaceInfo& InterfaceInfo : SortedInterfaces)
    {
        if (CompareVersionStrings(InterfaceInfo.Version, CurrentInterfaceVersion) < 0)
        {
            continue;
        }

        UUpdateListViewItem* Item = NewObject<UUpdateListViewItem>(this);

        UManagerInterfaceInfoWrapper* InterfaceWrapper = NewObject<UManagerInterfaceInfoWrapper>(this);
        InterfaceWrapper->Initialize(InterfaceInfo);

        TArray<FString> AndroidVersions = GetFilteredPlatformVersions(
            TEXT("Android"),
            InterfaceInfo.Version,
            CurrentVersionMap[TEXT("Android")]
        );

        TArray<FString> iOSVersions = GetFilteredPlatformVersions(
            TEXT("iOS"),
            InterfaceInfo.Version,
            CurrentVersionMap[TEXT("iOS")]
        );

        TArray<FString> WindowsVersions = GetFilteredPlatformVersions(
            TEXT("Windows"),
            InterfaceInfo.Version,
            CurrentVersionMap[TEXT("Windows")]
        );

        TArray<FString> MacVersions = GetFilteredPlatformVersions(
            TEXT("Mac"),
            InterfaceInfo.Version,
            CurrentVersionMap[TEXT("Mac")]
        );

        Item->Initialize(
            InterfaceWrapper,
            AndroidVersions,
            iOSVersions,
            MacVersions,
            WindowsVersions,
            CurrentVersionMap
        );

        Items.Add(Item);
    }

    UpdateListViewRawData->Initialize(Items, CurrentVersionMap, LatestVersionMap);
}

TArray<FString> UHiveSDKManagerSubsystem::GetIgnoredFileNames()
{
    return FHiveManagerIgnorePatchFileManager::Get().GetIgnoredFileNames();
}

void UHiveSDKManagerSubsystem::InstallOrUpdate(const TMap<FString, FString>& SelectedVersions)
{
    HM_LOG(TEXT("Called InstallOrUpdate()"));

    TArray<FString> Types = { TEXT("Android"), TEXT("iOS"), TEXT("Windows"), TEXT("Mac"), TEXT("Interface") };
    TArray<FString> TypesToRemove;

    // to remove and install
    for (const FString& Type : Types)
    {
        const FString* SelectedVersion = SelectedVersions.Find(Type);
        if (SelectedVersion)
        {
            FString CurrentVersion = GetCurrentVersion(Type);
            if (CompareVersionStrings(*SelectedVersion, CurrentVersion) > 0)
            {
                TypesToRemove.Add(Type);
            }
        }
    }

    if (TypesToRemove.IsEmpty())
    {
        return;
    }

    TMap<FString, FString> FilteredVersions;
    for (const FString& Type : TypesToRemove)
    {
        if (const FString* Version = SelectedVersions.Find(Type))
        {
            FilteredVersions.Add(Type, *Version);
        }
    }

    TSharedRef<int32> CompletedCount = MakeShared<int32>(0);
    int32 TotalCount = TypesToRemove.Num();

    for (const FString& Type : TypesToRemove)
    {
        if (Type.Equals(TEXT("Interface"), ESearchCase::IgnoreCase))
        {
            RemoveInterface(FOnDeleteComplete::CreateLambda(
                [this, CompletedCount, TotalCount, FilteredVersions](bool bSucceeded, const FString& ErrorMessage)
                {
                    (*CompletedCount)++;

                    if (!bSucceeded)
                    {
                        HM_LOG_WARNING(TEXT("Failed to remove interface: %s"), *ErrorMessage);
                    }

                    if (*CompletedCount >= TotalCount)
                    {
                        HM_LOG(TEXT("All types removed, proceeding to install"));
                        ProceedToInstall(FilteredVersions);
                    }
                }
            ));
        }
        else
        {
            RemovePlatform(Type, FOnDeleteComplete::CreateLambda(
                [this, CompletedCount, TotalCount, FilteredVersions](bool bSucceeded, const FString& ErrorMessage)
                {
                    (*CompletedCount)++;

                    if (!bSucceeded)
                    {
                        HM_LOG_WARNING(TEXT("Failed to remove platform: %s"), *ErrorMessage);
                    }

                    if (*CompletedCount >= TotalCount)
                    {
                        HM_LOG(TEXT("All types removed, proceeding to install"));
                        ProceedToInstall(FilteredVersions);
                    }
                }
            ));
        }
        
    }
}

void UHiveSDKManagerSubsystem::RemoveInterface(FOnDeleteComplete OnComplete)
{
    // Query interface details based on the current version
    FHiveManagerProtocolApi::RequestInterfaceDetail(
        GetCurrentVersion(TEXT("interface")),
        [OnComplete](bool bSucceeded, FManagerInterfaceDetailInfo ResponseData, const FString& ErrorMessage)
        {
            if (!bSucceeded) {
                HM_LOG_WARNING(TEXT("Failed RemoveInterface"));
                OnComplete.ExecuteIfBound(false, TEXT(""));
                return;
            }

            if (ResponseData.Files.IsEmpty()) {
                HM_LOG_WARNING(TEXT("Failed RemoveInterface"));
                OnComplete.ExecuteIfBound(false, TEXT(""));
                return;
            }

            TSet<FString> RootDirs;
            for (const FFile& File : ResponseData.Files)
            {
                FString ParentDir = FPaths::GetPath(File.Path);
                RootDirs.Add(ParentDir);
            }

            for (const FString& RootDir : RootDirs)
            {
                FString FullPath = FPaths::ProjectPluginsDir() / RootDir;
                if (IFileManager::Get().DirectoryExists(*FullPath))
                {
                    IFileManager::Get().DeleteDirectory(*FullPath, false, true);
                    HM_LOG(TEXT("Remove Dir : %s"), *RootDir);
                }
            }
            OnComplete.ExecuteIfBound(true, TEXT("Succeeded"));
        }
    );
}

void UHiveSDKManagerSubsystem::RemovePlatform(const FString& Platform, FOnDeleteComplete OnComplete)
{
    // Query platform details based on the current version
    FHiveManagerProtocolApi::RequestPlatformDetail(
        Platform.ToLower(),
        GetCurrentVersion(Platform),
        [OnComplete](bool bSucceeded, FManagerPlatformDetailInfo ResponseData, const FString& ErrorMessage)
        {
            if (!bSucceeded) {
                HM_LOG_WARNING(TEXT("Failed RemovePlatform"));
                OnComplete.ExecuteIfBound(false, TEXT(""));
                return;
            }

            if (ResponseData.Files.IsEmpty()) {
                HM_LOG_WARNING(TEXT("Failed RemovePlatform"));
                OnComplete.ExecuteIfBound(false, TEXT(""));
                return;
            }

            TSet<FString> RootDirs;
            for (const FFile& File : ResponseData.Files)
            {
                FString ParentDir = FPaths::GetPath(File.Path);
                RootDirs.Add(ParentDir);
            }

            for (const FString& RootDir : RootDirs)
            {
                FString FullPath = FPaths::ProjectPluginsDir() / RootDir;
                if (IFileManager::Get().DirectoryExists(*FullPath))
                {
                    IFileManager::Get().DeleteDirectory(*FullPath, false, true);
                    HM_LOG(TEXT("RemovePlatform : %s"), *RootDir);
                }
            }
            OnComplete.ExecuteIfBound(true, TEXT("Succeeded"));
        }
    );
}



void UHiveSDKManagerSubsystem::ProceedToInstall(const TMap<FString, FString>& SelectedVersions)
{
    UE_LOG(LogTemp, Log, TEXT("Called ProceedToInstall"));

    TWeakObjectPtr<UHiveSDKManagerSubsystem> WeakThis(this);
    const FString* InterfaceVersion = SelectedVersions.Find(TEXT("Interface"));

    if (InterfaceVersion)
    {
        InstallInterface(*InterfaceVersion, FOnInstallComplete::CreateLambda([SelectedVersions, WeakThis](bool bSucceeded, const FString& ErrorMessage)
            {
                if (!bSucceeded)
                {
                    if (WeakThis.IsValid())
                    {
                        WeakThis->OnInstallOrUpdateCompleteDelegate.Broadcast();
                    }
                }
                
                if (WeakThis.IsValid())
                {
                    TMap<FString, FString> PlatformVersions = SelectedVersions;
                    PlatformVersions.Remove(TEXT("Interface"));

                    WeakThis->InstallPlatform(PlatformVersions, FOnInstallComplete::CreateLambda([SelectedVersions, WeakThis](bool bSucceeded, const FString& ErrorMessage)
                        {
                            if (WeakThis.IsValid())
                            {
                                WeakThis->OnInstallOrUpdateCompleteDelegate.Broadcast();
                            }
                        }
                    ));
                }
            }
        ));
    }
    else
    {
        if (WeakThis.IsValid())
        {
            TMap<FString, FString> PlatformVersions = SelectedVersions;
            PlatformVersions.Remove(TEXT("Interface"));

            WeakThis->InstallPlatform(PlatformVersions, FOnInstallComplete::CreateLambda([SelectedVersions, WeakThis](bool bSucceeded, const FString& ErrorMessage)
                {
                    HM_LOG(TEXT("Called InstallInterface (FINISHED!!!)"));
                    if (WeakThis.IsValid())
                    {
                        WeakThis->OnInstallOrUpdateCompleteDelegate.Broadcast();
                    }
                }
            ));
        }
    }
}

void UHiveSDKManagerSubsystem::InstallInterface(const FString& Versions, FOnInstallComplete OnComplete)
{
    HM_LOG(TEXT("Called InstallInterface"));

    FHiveManagerProtocolApi::RequestInterfaceDetail(Versions, [Versions, OnComplete](bool bSucceeded, FManagerInterfaceDetailInfo ResponseData, const FString& ErrorMessage)
        {
            if (!bSucceeded || ResponseData.DownloadUrl.IsEmpty())
            {
                HM_LOG_WARNING(TEXT("Failed InstallInterface"));
                OnComplete.ExecuteIfBound(false, TEXT("Failed InstallInterface"));
                return;
            }

            FString DownloadPath = FPaths::ProjectSavedDir() / TEXT("HiveSDKManager") / TEXT("Download");

            IPlatformFile& PlatformFile = FPlatformFileManager::Get().GetPlatformFile();
            if (PlatformFile.DirectoryExists(*DownloadPath))
            {
                PlatformFile.DeleteDirectoryRecursively(*DownloadPath);
            }

            FScopedSlowTask PreprocessingSlowTask(3.f, FText::FromString("Preprocessing: Downloading, Unzipping..."));
            PreprocessingSlowTask.MakeDialog();
            FPlatformProcess::Sleep(1.f);

            HiveManagerFile::DownloadFile(
                ResponseData.DownloadUrl,
                DownloadPath,
                FDownloadProgressDelegate::CreateLambda([](uint64 BytesReceived) {}),
                FDownloadCompleteDelegate::CreateLambda([Versions, OnComplete](bool bSucceeded, const FString& DownloadFilePath)
                    {
                        if (!bSucceeded)
                        {
                            HM_LOG_WARNING(TEXT("Failed InstallInterface(DownloadFile)"));
                            OnComplete.ExecuteIfBound(false, TEXT("Failed InstallInterface(DownloadFile)"));
                            return;
                        }

                        // unzip
                        FString UnzipPath = FPaths::ProjectPluginsDir();

                        if (!HiveManagerFile::UnZipp(DownloadFilePath, UnzipPath))
                        {
                            HM_LOG_WARNING(TEXT("Failed InstallInterface(UnZipp)"));
                            OnComplete.ExecuteIfBound(false, TEXT("Failed InstallInterface(UnZipp)"));
                            return;
                        }

                        HM_LOG(TEXT("InstallInterface Succeeded"));

                        FHiveManagerProtocolApi::RequestManagerDownloadLogs(
                            TEXT("interface"),
                            Versions,
                            [OnComplete](bool bLogSucceeded, FManagerEmptyResponse ResponseData, const FString& LogErrorMessage)
                            {
                                if (!bLogSucceeded)
                                {
                                    HM_LOG_WARNING(TEXT("Failed to send download log for Interface"));
                                }
                                OnComplete.ExecuteIfBound(true, TEXT("Succeeded"));
                            }
                        );
                    }
                )
            );
        }
    );
}

void UHiveSDKManagerSubsystem::InstallPlatform(const TMap<FString, FString>& SelectedVersions, FOnInstallComplete OnComplete)
{
    struct FInstallPlatformState
    {
        int32 CompletedCount = 0;
        int32 TotalCount = 0;
        bool bHasFailed = false;
        FCriticalSection Mutex;
    };

    TSharedPtr<FInstallPlatformState> SharedState = MakeShared<FInstallPlatformState>();
    SharedState->TotalCount = SelectedVersions.Num();

    auto CheckAndExecuteCallback = [SharedState, OnComplete](bool bSuccess)
        {
            FScopeLock Lock(&SharedState->Mutex);
            SharedState->CompletedCount++;

            if (!bSuccess)
            {
                SharedState->bHasFailed = true;
            }

            if (SharedState->CompletedCount >= SharedState->TotalCount)
            {
                if (SharedState->bHasFailed)
                {
                    OnComplete.ExecuteIfBound(false, TEXT("InstallPlatform One or more tasks failed"));
                }
                else
                {
                    OnComplete.ExecuteIfBound(true, TEXT("InstallPlatform Succeeded"));
                }
            }
        };

    for (const TPair<FString, FString>& Pair : SelectedVersions)
    {
        const FString& Platform = Pair.Key;
        const FString& Version = Pair.Value;

        HM_LOG(TEXT("Called InstallPlatform(Platform: %s, Version: %s)"), *Platform, *Version);

        FHiveManagerProtocolApi::RequestPlatformDetail(Platform, Version, [CheckAndExecuteCallback, Platform, Version](bool bSucceeded, FManagerPlatformDetailInfo ResponseData, const FString& ErrorMessage)
            {
                if (!bSucceeded || ResponseData.DownloadUrl.IsEmpty())
                {
                    HM_LOG_WARNING(TEXT("Failed InstallPlatform"));
                    CheckAndExecuteCallback(false);
                    return;
                }

                FString DownloadPath = FPaths::ProjectSavedDir() / TEXT("HiveSDKManager") / TEXT("Download") / Platform /*for TEST*/;

                IPlatformFile& PlatformFile = FPlatformFileManager::Get().GetPlatformFile();
                if (PlatformFile.DirectoryExists(*DownloadPath))
                {
                    PlatformFile.DeleteDirectoryRecursively(*DownloadPath);
                }

                FScopedSlowTask PreprocessingSlowTask(3.f, FText::FromString("Preprocessing: Downloading, Unzipping..."));
                PreprocessingSlowTask.MakeDialog();
                FPlatformProcess::Sleep(1.f);

                HiveManagerFile::DownloadFile(
                    ResponseData.DownloadUrl,
                    DownloadPath,
                    FDownloadProgressDelegate::CreateLambda([](uint64 BytesReceived) {}),
                    FDownloadCompleteDelegate::CreateLambda([CheckAndExecuteCallback, Platform, Version](bool bSucceeded, const FString& DownloadFilePath)
                        {
                            if (!bSucceeded)
                            {
                                HM_LOG_WARNING(TEXT("Failed InstallPlatform(DownloadFile)"));
                                CheckAndExecuteCallback(false);
                                return;
                            }

                            // unzip
                            FString UnzipPath = FPaths::ProjectPluginsDir();

                            HM_LOG(TEXT("Unzip from %s to %s"), *DownloadFilePath, *UnzipPath);


                            if (!HiveManagerFile::UnZipp(DownloadFilePath, UnzipPath))
                            {
                                HM_LOG_WARNING(TEXT("Failed InstallPlatform(UnZipp)"));
                                CheckAndExecuteCallback(false);
                                return;
                            }

                            HM_LOG(TEXT("InstallPlatform Succeeded"));
                            
                            FHiveManagerProtocolApi::RequestManagerDownloadLogs(
                                Platform,
                                Version,
                                [CheckAndExecuteCallback](bool bLogSucceeded, FManagerEmptyResponse ResponseData, const FString& LogErrorMessage)
                                {
                                    if (!bLogSucceeded)
                                    {
                                        HM_LOG_WARNING(TEXT("Failed to send download log for Platform"));
                                    }
                                    CheckAndExecuteCallback(true);
                                }
                            );
                        }
                    ));
            });
    }
}

// 헬퍼 함수: 플랫폼별 필터링된 버전 가져오기
TArray<FString> UHiveSDKManagerSubsystem::GetFilteredPlatformVersions(
    const FString& Platform,
    const FString& InterfaceVersion,
    const FString& CurrentPlatformVersion)
{
    TArray<FString> FilteredVersions;

    // 해당 플랫폼의 캐시된 정보 가져오기
    TArray<FManagerPlatformInfo>* PlatformCache = nullptr;

    if (Platform == TEXT("Android"))
    {
        PlatformCache = &CachedAvailableAndroid;
    }
    else if (Platform == TEXT("iOS"))
    {
        PlatformCache = &CachedAvailableiOS;
    }
    else if (Platform == TEXT("Windows"))
    {
        PlatformCache = &CachedAvailableWindows;
    }
    else if (Platform == TEXT("Mac"))
    {
        PlatformCache = &CachedAvailableMac;
    }

    if (PlatformCache == nullptr)
    {
        return FilteredVersions;
    }

    // 조건에 맞는 버전만 필터링
    for (const FManagerPlatformInfo& PlatformInfo : *PlatformCache)
    {
        HM_LOG_WARNING(TEXT("PlatformInfo.Version = %s, InterfaceVersion = %s, CurrentPlatformVersion = %s"), *PlatformInfo.Version, * InterfaceVersion, * CurrentPlatformVersion);


        // 조건 1: 인터페이스 버전과 a.b는 같고 c만 높아야 함
        bool bMatchesInterface = IsPatchVersionHigher(PlatformInfo.Version, InterfaceVersion);

        // 조건 2: 현재 플랫폼 버전과 같거나 높아야 함
        bool bMatchesCurrent = CompareVersionStrings(PlatformInfo.Version, CurrentPlatformVersion) >= 0;

        // 둘 중 하나라도 만족하면 포함
        if (bMatchesInterface && bMatchesCurrent)
        {
            FilteredVersions.Add(PlatformInfo.Version);
        }
    }

    // 내림차순 정렬
    FilteredVersions.Sort([this](const FString& A, const FString& B) {
        return CompareVersionStrings(A, B) > 0;
        });

    return FilteredVersions;
}

// 헬퍼 함수: a.b.c 형식에서 a, b는 같고 c만 높은지 확인
bool UHiveSDKManagerSubsystem::IsPatchVersionHigher(const FString& VersionA, const FString& BaseVersion)
{
    TArray<FString> PartsA;
    TArray<FString> PartsBase;

    VersionA.ParseIntoArray(PartsA, TEXT("."));
    BaseVersion.ParseIntoArray(PartsBase, TEXT("."));

    // 최소 3개의 파트가 있어야 함 (a.b.c)
    if (PartsA.Num() < 3 || PartsBase.Num() < 3)
    {
        return false;
    }

    int32 MajorA = FCString::Atoi(*PartsA[0]);
    int32 MinorA = FCString::Atoi(*PartsA[1]);
    int32 PatchA = FCString::Atoi(*PartsA[2]);

    int32 MajorBase = FCString::Atoi(*PartsBase[0]);
    int32 MinorBase = FCString::Atoi(*PartsBase[1]);
    int32 PatchBase = FCString::Atoi(*PartsBase[2]);

    // a, b는 같고 c만 높아야 함
    return (MajorA == MajorBase) && (MinorA == MinorBase) && (PatchA >= PatchBase);
}

void UHiveSDKManagerSubsystem::RestartEditor()
{
    FUnrealEdMisc::Get().RestartEditor();
}