/* Copyright © 2025-Present Com2uS Platform Corp. All Rights Reserved. */

#include "HiveManagerIgnorePathFileManager.h"
#include "Logger/HiveManagerLogger.h"

// Static 멤버 초기화
const FString FHiveManagerIgnorePatchFileManager::IgnoreFileName = TEXT("HiveSDKManagerIgnore.txt");
const FString FHiveManagerIgnorePatchFileManager::IgnoreFilePath = FPaths::Combine(FPaths::Combine(FPaths::ProjectPluginsDir(), TEXT("HiveSDKManager")), IgnoreFileName);

const TArray<FString> FHiveManagerIgnorePatchFileManager::ImportantFileList = {
	// example
	// TEXT("Hive_SDK_v4/Plugins/Android/libs/HIVE_SDK.aar"),
};

const FString FHiveManagerIgnorePatchFileManager::CommentIgnoreFile = TEXT(
	"# Hive SDK Manager Ignore file\n"
	"# Enter the folder or directory path to be excluded when patching for Hive SDK Manager.\n"
	"# The path entered in this file is ignored when patching for Hive SDK Manager\n"
	"# Enter the path under Assets as the relative path.\n"
	"# Folder ignore example\n"
	"# foo/bar\n"
	"# File ignore example\n"
	"# foo/bar.txt"
);

FHiveManagerIgnorePatchFileManager::FHiveManagerIgnorePatchFileManager()
	: IgnorePatchFile(false)
	, IgnoreFilter(false)
{
}

FHiveManagerIgnorePatchFileManager& FHiveManagerIgnorePatchFileManager::Get()
{
	static FHiveManagerIgnorePatchFileManager Instance;
	return Instance;
}

bool FHiveManagerIgnorePatchFileManager::IsCaseSensitiveSystem() const
{
	IPlatformFile& PlatformFile = FPlatformFileManager::Get().GetPlatformFile();

	FString TempPath = FPaths::CreateTempFilename(*FPaths::ProjectSavedDir(), TEXT("CaseTest_"));
	FString LowerCasePath = TempPath.ToLower();
	FString UpperCasePath = TempPath.ToUpper();

	// 소문자 파일 생성
	TUniquePtr<IFileHandle> FileHandle(PlatformFile.OpenWrite(*LowerCasePath));
	if (FileHandle)
	{
		FileHandle->Flush();
		FileHandle.Reset();
	}

	// 대문자 경로로 존재 여부 확인
	bool bIsCaseSensitive = !PlatformFile.FileExists(*UpperCasePath);

	// 파일 삭제
	PlatformFile.DeleteFile(*LowerCasePath);

	return bIsCaseSensitive;
}

void FHiveManagerIgnorePatchFileManager::LoadIgnore()
{
	bool bIsCaseSensitive = IsCaseSensitiveSystem();

	IgnorePatchFile = FHiveManagerIgnorePatchFile(IgnoreFilePath, bIsCaseSensitive);
	IgnorePatchFile.Parse();

	IgnoreFilter = FHiveManagerIgnorePatchFile(bIsCaseSensitive);
	for (int32 i = 0; i < ImportantFileList.Num(); ++i)
	{
		IgnoreFilter.InsertIgnorePath(ImportantFileList[i]);
	}
}

bool FHiveManagerIgnorePatchFileManager::IsIgnore(const FString& Ignore) const
{
	// IgnoreFilter 목록에 걸리면 중요 파일이므로 ignore 검사를 하지 않고 return false를 무시목록에서 제외
	if (IgnoreFilter.IsIgnore(Ignore))
	{
		return false;
	}
	return IgnorePatchFile.IsIgnore(Ignore);
}

void FHiveManagerIgnorePatchFileManager::EditIgnoreFile()
{

	IPlatformFile& PlatformFile = FPlatformFileManager::Get().GetPlatformFile();

	if (!PlatformFile.FileExists(*IgnoreFilePath))
	{
		FFileHelper::SaveStringToFile(CommentIgnoreFile, *IgnoreFilePath);
	}

	FString IgnoreFilePathAbs = FPaths::ConvertRelativePathToFull(IgnoreFilePath);

	// 파일을 기본 에디터로 열기
	if (!FPlatformProcess::LaunchFileInDefaultExternalApplication(*IgnoreFilePathAbs, nullptr, ELaunchVerb::Open))
	{
		HM_LOG_WARNING(TEXT("Failed LaunchFileInDefaultExternalApplication.\nPATH = %s"), *IgnoreFilePathAbs);
	}
	
}

TArray<FString> FHiveManagerIgnorePatchFileManager::GetIgnoredFileNames()
{
	TArray<FString> Result;

	IPlatformFile& PlatformFile = FPlatformFileManager::Get().GetPlatformFile();

	if (!PlatformFile.FileExists(*IgnoreFilePath))
	{
		FFileHelper::SaveStringToFile(CommentIgnoreFile, *IgnoreFilePath);
	}

	TArray<FString> Lines;
	if (FFileHelper::LoadFileToStringArray(Lines, *IgnoreFilePath))
	{
		for (const FString& Line : Lines)
		{
			FString TrimmedLine = Line.TrimStartAndEnd();

			// # 로 시작하지 않고 비어있지 않은 라인만 추가
			if (!TrimmedLine.StartsWith(TEXT("#")) && !TrimmedLine.IsEmpty())
			{
				Result.Add(TrimmedLine);
			}
		}
	}

	return Result;
}
