/* Copyright © 2025-Present Com2uS Platform Corp. All Rights Reserved. */

#include "HiveManagerIgnorePathTree.h"

void FHiveManagerPathTree::Insert(const TArray<FString>& Paths, int32 CurrentIndex)
{
	bool bFind = false;
	bool bIsLastPath = (Paths.Num() - 1 == CurrentIndex);
	const FString& Path = Paths[CurrentIndex];

	for (int32 i = 0; i < Children.Num(); ++i)
	{
		if (Children[i].Data == Path)
		{
			if (bIsLastPath)
			{
				// 마지막인데 자식이 존재할 경우 자식들을 모두 제거해준다.
				// ex. 기존 /A/B/C가 있는데 /A/B를 삽입하려는 경우, 기존 /A/B/C에서 /C를 유지할 필요 없음
				if (Children[i].Children.Num() > 0)
				{
					Children[i].Children.Empty();
				}
			}
			else
			{
				// 마지막 패스일 경우 더이상 추가하지 않는다.
				// ex. 기존 /A/B가 있는데 /A/B/C를 삽입하려는 경우, /B노드 체크에서 /C 자식 노드를 넣을 필요 없음
				// ex. 기존 /A/B/? 가 있는데 /A/B/C를 삽입하려는 경우, /B노드 체크 이후 /C 자식 노드 삽입 일단 시도
				if (Children[i].Children.Num() > 0)
				{
					Children[i].Insert(Paths, CurrentIndex + 1);
				}
			}
			bFind = true;
			break;
		}
	}

	if (!bFind)
	{
		// 새로운 자식 추가
		FHiveManagerPathTree& NewChild = Children.AddDefaulted_GetRef();
		NewChild.Data = Path;

		if (!bIsLastPath)
		{
			// 마지막이 아니라면 하위노드들도 삽입될 수 있도록
			// ex. 기존 /B가 있는데 /A 를 삽입 하려는 경우, /A 삽입 및 종료
			// ex. 기존 /B가 있는데 /A/B를 삽입 하려는 경우, /A 삽입 이후 자식 노드 /B 까지 추가될 수 있도록 수행
			NewChild.Insert(Paths, CurrentIndex + 1);
		}
	}
}

FString FHiveManagerPathTree::ChangePathWindowsToUnix(const FString& Path) const
{
	return Path.Replace(TEXT("\\"), TEXT("/"));
}

void FHiveManagerPathTree::Insert(const FString& Path)
{
	FString UnixPath = ChangePathWindowsToUnix(Path);
	TArray<FString> Paths;
	UnixPath.ParseIntoArray(Paths, TEXT("/"));
	Insert(Paths, 0);
}

bool FHiveManagerPathTree::IsIgnore(const TArray<FString>& Paths, int32 CurrentIndex) const
{
	const FString& Path = Paths[CurrentIndex];
	bool bIsLastPath = (Paths.Num() - 1 == CurrentIndex);

	for (int32 i = 0; i < Children.Num(); ++i)
	{
		if (Children[i].Data == Path)
		{
			if (Children[i].Children.Num() == 0)
			{
				// 하위패스가 존재하지 않으므로 무시가능
				// ex. 기존 /A 가 등록된 상태에서 /A/B 는 무시 가능함
				return true;
			}
			else
			{
				if (bIsLastPath)
				{
					// 문자열 마지막에 하위노드 존재시 ignore 대상 아님
					// ex. 기존 /A/B 가 등록된 상태에서 /A 비교시, 대상 아님
					return false;
				}
				else
				{
					// 다음패스로 진행
					return Children[i].IsIgnore(Paths, CurrentIndex + 1);
				}
			}
		}
	}

	return false;
}

bool FHiveManagerPathTree::IsIgnore(const FString& Path) const
{
	FString UnixPath = ChangePathWindowsToUnix(Path);
	TArray<FString> Paths;
	UnixPath.ParseIntoArray(Paths, TEXT("/"));
	return IsIgnore(Paths, 0);
}