/* Copyright © 2025-Present Com2uS Platform Corp. All Rights Reserved. */

#include "Localization/HiveManagerStrings.h"
#include "Logger/HiveManagerLogger.h"
#include "Internationalization/Internationalization.h"
#include "Internationalization/Culture.h"

UHiveManagerStrings::UHiveManagerStrings()
{
}

void UHiveManagerStrings::Initialize()
{
    FString MetaFilePath = FPaths::ProjectPluginsDir() + TEXT("HiveSDKManager/Resources/HiveManagerStrings.json");
    LoadFromJsonFile(MetaFilePath);
}

FString UHiveManagerStrings::GetLocalizedText(const FString& Key) const
{
    const TMap<FString, FString>* LanguageMap = LocalizationData.Find(Key);
    if (LanguageMap)
    {
        const FString* Text = LanguageMap->Find(GetCurrentEditorLanguage());
        if (Text)
        {
            // Converts '\\n' to an actual newline character
            FString Result = *Text;
            Result = Result.Replace(TEXT("\\n"), TEXT("\n"));
            return Result;
        }
    }

    HM_LOG_WARNING(TEXT("No data was found for the given key. key = %s"), *Key);

    return TEXT("");
}

bool UHiveManagerStrings::LoadFromJsonFile(const FString& FilePath)
{
    FString JsonString;
    if (!FFileHelper::LoadFileToString(JsonString, *FilePath))
    {
        HM_LOG_WARNING(TEXT("Failed to load JSON file: %s"), * FilePath);
        return false;
    }

    return LoadFromJsonString(JsonString);
}

bool UHiveManagerStrings::LoadFromJsonString(const FString& JsonString)
{
	TSharedPtr<FJsonObject> JsonObject;
	TSharedRef<TJsonReader<>> Reader = TJsonReaderFactory<>::Create(JsonString);

    if (!FJsonSerializer::Deserialize(Reader, JsonObject) || !JsonObject.IsValid())
    {
        HM_LOG_WARNING(TEXT("Failed to parse JSON string"));
        return false;
    }

    LocalizationData.Empty();

    for (const auto& Pair : JsonObject->Values)
    {
        const FString& Key = Pair.Key;
        TSharedPtr<FJsonObject> LanguageObject = Pair.Value->AsObject();

        if (LanguageObject.IsValid())
        {
            TMap<FString, FString> LanguageTexts;

            for (const auto& LangPair : LanguageObject->Values)
            {
                const FString& LangCode = LangPair.Key;
                FString Text;
                if (LangPair.Value->TryGetString(Text))
                {
                    LanguageTexts.Add(LangCode, Text);
                }
            }

            LocalizationData.Add(Key, LanguageTexts);
        }
    }

    HM_LOG(TEXT("Successfully loaded %d localization keys"), LocalizationData.Num());
    return true;
}

FString UHiveManagerStrings::GetCurrentEditorLanguage() const
{
    FString LanguageCode = FInternationalization::Get().GetCurrentCulture()->GetTwoLetterISOLanguageName().ToLower();
    
    // only support "en", "ko", "ja"
    if (LanguageCode == TEXT("en") ||
        LanguageCode == TEXT("ko") ||
        LanguageCode == TEXT("ja"))
    {
        return LanguageCode;
    }

    return TEXT("en");
}