/* Copyright © 2025-Present Com2uS Platform Corp. All Rights Reserved. */

#include "HiveManagerProtocolApi.h"

#include "HiveSDKManagerConfig.h"
#include "HiveManagerProtocolUrl.h"

FHiveManagerProtocolApi::FHiveManagerProtocolApi()
{
}

FHiveManagerProtocolApi::~FHiveManagerProtocolApi()
{
}

void FHiveManagerProtocolApi::RequestInterfaceAll(FResponseCallback<TArray<FManagerInterfaceInfo>> Callback)
{
	TSharedPtr<FJsonObject> JsonBody = MakeShared<FJsonObject>();
	JsonBody->SetStringField(TEXT("engine"), UTF8_TO_TCHAR(HiveConfig::Engine));
	JsonBody->SetStringField(TEXT("language"), FHiveManagerProtocolApi::GetCurrentEditorLanguage());

	FHiveManagerProtocol::RequestPost<TArray<FManagerInterfaceInfo>>(
		FHiveManagerProtocolUrl::Get(HiveConfig::bIsTest) + TEXT("/interface/all"),
		JsonBody,
		Callback
	);
}

void FHiveManagerProtocolApi::RequestInterfaceDetail(const FString& Version, FResponseCallback<FManagerInterfaceDetailInfo> Callback)
{
	TSharedPtr<FJsonObject> JsonBody = MakeShared<FJsonObject>();
	JsonBody->SetStringField(TEXT("engine"), UTF8_TO_TCHAR(HiveConfig::Engine));
	JsonBody->SetStringField(TEXT("language"), FHiveManagerProtocolApi::GetCurrentEditorLanguage());
	JsonBody->SetStringField(TEXT("version"), Version);

	FHiveManagerProtocol::RequestPost<FManagerInterfaceDetailInfo>(
		FHiveManagerProtocolUrl::Get(HiveConfig::bIsTest) + TEXT("/interface/detail"),
		JsonBody,
		Callback
	);
}

void FHiveManagerProtocolApi::RequestInterfaceCompare(const FString& BaseVersion, const FString& CompareVersion, FResponseCallback<FManagerCompareInfo> Callback)
{
	TSharedPtr<FJsonObject> JsonBody = MakeShared<FJsonObject>();
	JsonBody->SetStringField(TEXT("engine"), UTF8_TO_TCHAR(HiveConfig::Engine));
	JsonBody->SetStringField(TEXT("baseVersion"), BaseVersion);
	JsonBody->SetStringField(TEXT("compareVersion"), CompareVersion);

	FHiveManagerProtocol::RequestPost<FManagerCompareInfo>(
		FHiveManagerProtocolUrl::Get(HiveConfig::bIsTest) + TEXT("/interface/compare"),
		JsonBody,
		Callback
	);
}

void FHiveManagerProtocolApi::RequestPlatformAll(const FString& Platform, FResponseCallback<TArray<FManagerPlatformInfo>> Callback)
{
	TSharedPtr<FJsonObject> JsonBody = MakeShared<FJsonObject>();
	JsonBody->SetStringField(TEXT("engine"), UTF8_TO_TCHAR(HiveConfig::Engine));
	JsonBody->SetStringField(TEXT("platform"), Platform);

	FHiveManagerProtocol::RequestPost<TArray<FManagerPlatformInfo>>(
		FHiveManagerProtocolUrl::Get(HiveConfig::bIsTest) + TEXT("/platform/all"),
		JsonBody,
		Callback
	);
}

void FHiveManagerProtocolApi::RequestPlatformDetail(const FString& Platform, const FString& Version, FResponseCallback<FManagerPlatformDetailInfo> Callback)
{
	TSharedPtr<FJsonObject> JsonBody = MakeShared<FJsonObject>();
	JsonBody->SetStringField(TEXT("engine"), UTF8_TO_TCHAR(HiveConfig::Engine));
	JsonBody->SetStringField(TEXT("platform"), Platform);
	JsonBody->SetStringField(TEXT("version"), Version);

	FHiveManagerProtocol::RequestPost<FManagerPlatformDetailInfo>(
		FHiveManagerProtocolUrl::Get(HiveConfig::bIsTest) + TEXT("/platform/detail"),
		JsonBody,
		Callback
	);
}

void FHiveManagerProtocolApi::RequestPlatformCompare(const FString& Platformm, const FString& BaseVersion, const FString& CompareVersion, FResponseCallback<FManagerCompareInfo> Callback)
{
	TSharedPtr<FJsonObject> JsonBody = MakeShared<FJsonObject>();
	JsonBody->SetStringField(TEXT("engine"), UTF8_TO_TCHAR(HiveConfig::Engine));
	JsonBody->SetStringField(TEXT("platform"), Platformm);
	JsonBody->SetStringField(TEXT("baseVersion"), BaseVersion);
	JsonBody->SetStringField(TEXT("compareVersion"), CompareVersion);

	FHiveManagerProtocol::RequestPost<FManagerCompareInfo>(
		FHiveManagerProtocolUrl::Get(HiveConfig::bIsTest) + TEXT("/platform/compare"),
		JsonBody,
		Callback
	);
}

void FHiveManagerProtocolApi::RequestNoticeInfo(FResponseCallback<TArray<FManagerNoticeInfo>> Callback)
{
	TSharedPtr<FJsonObject> JsonBody = MakeShared<FJsonObject>();
	JsonBody->SetStringField(TEXT("language"), FHiveManagerProtocolApi::GetCurrentEditorLanguage());

	FHiveManagerProtocol::RequestPost<TArray<FManagerNoticeInfo>>(
		FHiveManagerProtocolUrl::Get(HiveConfig::bIsTest) + TEXT("/notice/available"),
		JsonBody,
		Callback
	);
}

void FHiveManagerProtocolApi::RequestManagerDownloadLogs(const FString& Type, const FString& DownloadVersion, FResponseCallback<FManagerEmptyResponse> Callback)
{
	FString Data = FDateTime::Now().ToString(TEXT("%Y-%m-%d %H:%M:%S"));

	TSharedPtr<FJsonObject> JsonBody = MakeShared<FJsonObject>();
	JsonBody->SetStringField(TEXT("managerVersion"), UTF8_TO_TCHAR(HiveConfig::Version));
	JsonBody->SetStringField(TEXT("os"), FHiveManagerProtocolApi::GetHostOS());
	JsonBody->SetStringField(TEXT("engine"), UTF8_TO_TCHAR(HiveConfig::Engine));
	JsonBody->SetStringField(TEXT("platform"), Type);
	//JsonBody->SetStringField(TEXT("currentVersion"), TEXT("ko")); optional
	JsonBody->SetStringField(TEXT("downloadVersion"), DownloadVersion);
	JsonBody->SetStringField(TEXT("companyName"), FHiveManagerProtocolApi::GetCompanyName());
	JsonBody->SetStringField(TEXT("productName"), FApp::GetProjectName());
	JsonBody->SetStringField(TEXT("appid"), FHiveManagerProtocolApi::GetAppIdentifier()); // NOT SUPPORT UE
	JsonBody->SetStringField(TEXT("downloadDate"), Data);


	FString JsonString;

	TSharedRef<TJsonWriter<>> Writer = TJsonWriterFactory<>::Create(&JsonString);
	if (FJsonSerializer::Serialize(JsonBody.ToSharedRef(), Writer))
	{
		UE_LOG(LogTemp, Log, TEXT("JsonBody: %s"), *JsonString);
	}

	FHiveManagerProtocol::RequestPost<FManagerEmptyResponse>(
		FHiveManagerProtocolUrl::Get(HiveConfig::bIsTest) + TEXT("/manager/download-logs"),
		JsonBody,
		Callback
	);
}

void FHiveManagerProtocolApi::RequestManagerServerState(FResponseCallback<FManagerEmptyResponse> Callback)
{
	FHiveManagerProtocol::RequestGet<FManagerEmptyResponse>(
		FHiveManagerProtocolUrl::Get(HiveConfig::bIsTest) + TEXT("/check"),
		Callback
	);
}

FString FHiveManagerProtocolApi::GetCurrentEditorLanguage()
{
	FString LanguageCode = FInternationalization::Get().GetCurrentCulture()->GetTwoLetterISOLanguageName().ToLower();

	// only support "en", "ko", "ja"
	if (LanguageCode == TEXT("en") ||
		LanguageCode == TEXT("ko") ||
		LanguageCode == TEXT("ja"))
	{
		return LanguageCode;
	}

	return TEXT("en");
}

FString FHiveManagerProtocolApi::GetHostOS()
{
#if PLATFORM_WINDOWS
	return TEXT("Windows");
#elif PLATFORM_MAC
	return TEXT("mac");
#else
	return TEXT("Unknown");
#endif
}

FString FHiveManagerProtocolApi::GetCompanyName()
{
	FString CompanyName;
	GConfig->GetString(
		TEXT("/Script/EngineSettings.GeneralProjectSettings"),
		TEXT("CompanyName"),
		CompanyName,
		GGameIni
	);

	if (CompanyName.IsEmpty())
	{
		CompanyName = TEXT("DefaultCompany");
	}

	return CompanyName;
}

FString FHiveManagerProtocolApi::GetAppIdentifier()
{
	FString AppIdentifier;
	GConfig->GetString(
		TEXT("/Script/AndroidRuntimeSettings.AndroidRuntimeSettings"),
		TEXT("PackageName"),
		AppIdentifier,
		GEngineIni
	);

	
	if (!AppIdentifier.IsEmpty())
	{
		return AppIdentifier;
	}

	GConfig->GetString(
		TEXT("/Script/IOSRuntimeSettings.IOSRuntimeSettings"),
		TEXT("BundleIdentifier"),
		AppIdentifier,
		GEngineIni
	);

	if (!AppIdentifier.IsEmpty())
	{
		return AppIdentifier;
	}

	AppIdentifier = FApp::GetProjectName();

	if (!AppIdentifier.IsEmpty())
	{
		return AppIdentifier;
	}

	return TEXT("Default AppIdentifier");
}