/* Copyright © 2025-Present Com2uS Platform Corp. All Rights Reserved. */
#pragma once

#include "CoreMinimal.h"
#include "EditorSubsystem.h"

#include "Protocol/HiveManagerProtocolTypes.h"
#include "Localization/HiveManagerStrings.h"

#include "Models/ManagerNoticeInfoViewModel.h"
#include "Models/UpdateListViewModel.h"

#include "HiveSDKManagerSubsystem.generated.h"


DECLARE_DELEGATE_OneParam(FOnUpdateAvailableInterfaceCompleted, bool /*bAllSucceeded*/);
DECLARE_DELEGATE_OneParam(FOnUpdateAvailablePlatformCompleted, bool /*bAllSucceeded*/);

DECLARE_DYNAMIC_MULTICAST_DELEGATE_OneParam(FOnNoticeInfoReceived, const TArray<UManagerNoticeInfoViewModel*>&, NoticeList);
DECLARE_DELEGATE_OneParam(FOnIntegrityVerificationComplete, const TArray<FString>&);
DECLARE_DELEGATE_TwoParams(FOnIntegrityVerificationInterfaceComplete, const TArray<FString>& /* InvalidFiles */, const FManagerInterfaceDetailInfo&);
DECLARE_DELEGATE_TwoParams(FOnIntegrityVerificationPlatformComplete, const TArray<FString>& /* InvalidFiles */, const FManagerPlatformDetailInfo&);

DECLARE_DELEGATE_TwoParams(FOnApplyPatchComplete, bool /* bSucceeded*/, const FString& /* ErrorMessage */);
DECLARE_DELEGATE_TwoParams(FOnRestoreComplete, bool /* bSucceeded*/, const FString& /* ErrorMessage */);
DECLARE_DELEGATE_TwoParams(FOnDeleteComplete, bool /* bSucceeded*/, const FString& /* ErrorMessage */);
DECLARE_DELEGATE_TwoParams(FOnInstallComplete, bool /* bSucceeded*/, const FString& /* ErrorMessage */);

DECLARE_DYNAMIC_MULTICAST_DELEGATE_OneParam(FOnIntegrityVerificationCompleteDelegate, const TArray<FString>&, InvalidFiles);
DECLARE_DYNAMIC_MULTICAST_DELEGATE_OneParam(FOnRetryToConnectManagerServerDelegate, bool, bSucceeded);
DECLARE_DYNAMIC_MULTICAST_DELEGATE(FOnRestoreCompleteDelegate);
DECLARE_DYNAMIC_MULTICAST_DELEGATE(FOnInstallOrUpdateCompleteDelegate);


UCLASS()
class HIVESDKMANAGER_API UHiveSDKManagerSubsystem : public UEditorSubsystem
{
	GENERATED_BODY()
	
    // 에디터 시작 시 자동 호출
    virtual void Initialize(FSubsystemCollectionBase& Collection) override;

    // 에디터 종료 시 자동 호출
    virtual void Deinitialize() override;

public:
    FOnUpdateAvailableInterfaceCompleted OnUpdateAvailableInterfaceCompleted;
    FOnUpdateAvailablePlatformCompleted OnUpdateAvailablePlatformCompleted;

    UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "HiveSDKManager|UI")
    UHiveManagerStrings* HiveManagerStrings;

    UPROPERTY(BlueprintAssignable, Category = "HiveSDKManager")
    FOnNoticeInfoReceived OnNoticeInfoReceived;

    UPROPERTY(BlueprintAssignable, Category = "HiveSDKManager")
    FOnIntegrityVerificationCompleteDelegate OnIntegrityVerificationCompleteDelegate;

    UPROPERTY(BlueprintAssignable, Category = "HiveSDKManager")
    FOnRetryToConnectManagerServerDelegate OnRetryToConnectManagerServerDelegate;

    UPROPERTY(BlueprintAssignable, Category = "HiveSDKManager")
    FOnRestoreCompleteDelegate OnRestoreComplete;

    UPROPERTY(BlueprintAssignable, Category = "HiveSDKManager")
    FOnInstallOrUpdateCompleteDelegate OnInstallOrUpdateCompleteDelegate;

    UFUNCTION(BlueprintCallable, Category = "HiveSDKManager")
    FString GetCurrentVersion(const FString& Type);

    UFUNCTION(BlueprintCallable, Category = "HiveSDKManager")
    bool IsHiveSDKInstalled();

    UFUNCTION(BlueprintCallable, Category = "HiveSDKManager")
    bool IsInitialized();


    UFUNCTION(BlueprintCallable, Category = "HiveSDKManager|UI|Version")
    FString GetAllCurrentVersionForUI();

    UFUNCTION(BlueprintCallable, Category = "HiveSDKManager|UI|Version")
    FString GetAllLatestVersionForUI();

    UFUNCTION(BlueprintCallable, Category = "HiveSDKManager|UI|Version")
    FString GetManagerVersion();

    UFUNCTION(BlueprintCallable, Category = "HiveSDKManager|UI|Notice")
    void GetNotice();

    UFUNCTION(BlueprintCallable, Category = "HiveSDKManager|UI|IntegrityVerification")
    void IntegrityVerificationBP();

    UFUNCTION(BlueprintCallable, Category = "HiveSDKManager|UI|RetryToConnectManagerServer")
    void RetryToConnectManagerServer();

    UFUNCTION(BlueprintCallable, Category = "HiveSDKManager|UI|EditIgnorePatchFile")
    void EditIgnorePatchFile();

    UFUNCTION(BlueprintCallable, Category = "HiveSDKManager|UI|Restore")
    void Restore();

    UFUNCTION(BlueprintCallable, Category = "HiveSDKManager|UI|Update")
    UUpdateListViewModel* GetUpdateListViewData();

    UFUNCTION(BlueprintCallable, Category = "HiveSDKManager|UI|Update")
    void UpdateUpdateListViewData();

    UFUNCTION(BlueprintCallable, Category = "HiveSDKManager|UI|Update")
    TArray<FString> GetIgnoredFileNames();

    UFUNCTION(BlueprintCallable, Category = "HiveSDKManager|UI|Update")
    void InstallOrUpdate(const TMap<FString, FString>& SelectedVersions);

    UFUNCTION(BlueprintCallable, Category = "HiveSDKManager|UI|Update")
    void RestartEditor();

private:

    UUpdateListViewModel* UpdateListViewRawData;

    TArray<FManagerInterfaceInfo> CachedAvailableInterface;
    TArray<FManagerPlatformInfo> CachedAvailableAndroid;
    TArray<FManagerPlatformInfo> CachedAvailableiOS;
    TArray<FManagerPlatformInfo> CachedAvailableWindows;
    TArray<FManagerPlatformInfo> CachedAvailableMac;

    FString ReadVersionFromJson(const FString& FilePath);

    void UpdateAvailableInterface();
    void UpdateAvailablePlatform();
    void UpdateAvailablePlatform_Recursive(const TArray<FString>& Platforms, int32 Index, TWeakObjectPtr<UHiveSDKManagerSubsystem> WeakThis, bool bSoFarSucceeded);

    FManagerInterfaceInfo* GetLatestManagerInterfaceInfo();
    FString GetLatestAvailableInterfaceVersion();

    FManagerPlatformInfo* GetLatestManagerPlatformInfo(const FString& Platform);
    FString GetLatestAvailablePlatformVersion(const FString& Platform);

    void IntegrityVerificationInterface(FOnIntegrityVerificationInterfaceComplete OnComplete);
    void IntegrityVerificationPlatform(const FString& Platform, FOnIntegrityVerificationPlatformComplete OnComplete);

    void IntegrityVerification(FOnIntegrityVerificationComplete OnComplete);

    TArray<FString> FilterIgnoredFiles(const TArray<FString>& Files);
    void ApplyPatch(const FString& Type, const TArray<FString>& TargetFiles, const FString& DownloadUrl, FOnApplyPatchComplete OnComplete);

    void ResetoreInterface(FOnRestoreComplete OnComplete);
    void RestorePlatform(const FString& Platform, FOnRestoreComplete OnComplete);

    TArray<FString> GetFilteredPlatformVersions(
        const FString& Platform,
        const FString& InterfaceVersion,
        const FString& CurrentPlatformVersion);

    bool IsPatchVersionHigher(const FString& VersionA, const FString& BaseVersion);


    void RemovePlatform(const FString& Platform, FOnDeleteComplete OnComplete);
    void RemoveInterface(FOnDeleteComplete OnComplete);

    void ProceedToInstall(const TMap<FString, FString>& SelectedVersions);
    
    void InstallInterface(const FString& Versions, FOnInstallComplete OnComplete);
    void InstallPlatform(const TMap<FString, FString>& SelectedVersions, FOnInstallComplete OnComplete);

};
