/* Copyright © 2025-Present Com2uS Platform Corp. All Rights Reserved. */

#pragma once

#include "CoreMinimal.h"
#include "Http.h"
#include "Json.h"
#include "JsonUtilities.h"

template<typename T>
using FResponseCallback = TFunction<void(bool bSucceeded, T ResponseData, const FString& ErrorMessage)>;

class HIVESDKMANAGER_API FHiveManagerProtocol
{
public:
	FHiveManagerProtocol();
	~FHiveManagerProtocol();

	template<typename T>
	static void SendRequest(
		TSharedRef<IHttpRequest, ESPMode::ThreadSafe> Request,
		FResponseCallback<T> Callback
	);

	template<typename T>
	static void RequestGet(
		const FString& Url,
		FResponseCallback<T> Callback
	);

	template<typename T>
	static void RequestPost(
		const FString& Url,
		TSharedPtr<FJsonObject> JsonBody,
		FResponseCallback<T> Callback
	);

private:
	template<typename T>
	static bool ParseJsonStringToStruct(const FString& JsonString, T& OutStruct, FString& OutErrorMessage);
};

template<typename T>
void FHiveManagerProtocol::SendRequest(
	TSharedRef<IHttpRequest, ESPMode::ThreadSafe> Request,
	FResponseCallback<T> Callback
) {
	UE_LOG(LogTemp, Warning, TEXT("FHiveManagerProtocol::SendRequest() Called"));

	Request->SetHeader(TEXT("Content-Type"), TEXT("application/json;charset=utf-8"));

	Request->OnProcessRequestComplete().BindLambda([Callback](FHttpRequestPtr HttpRequest, FHttpResponsePtr HttpResponse, bool bSucceeded) {

		UE_LOG(LogTemp, Warning, TEXT("FHiveManagerProtocol::SendRequest() OnProcessRequestCompleted : %s"), bSucceeded ? TEXT("SUCCESS") : TEXT("failed"));

		T ResponseData;
		FString ErrorMessage;

		// handle network connection error
		if (!bSucceeded || !HttpResponse.IsValid())
		{
			ErrorMessage = FString::Printf(
				TEXT("HTTP Request Failed\nURL: %s\nbSucceeded = %s\nIsValid = %s"),
				*HttpRequest->GetURL(),
				bSucceeded ? TEXT("TRUE") : TEXT("FALSE"),
				HttpResponse.IsValid() ? TEXT("TRUE") : TEXT("FALSE")
			);

			UE_LOG(LogTemp, Warning, TEXT("%s"), *ErrorMessage);

			if (Callback)
			{
				Callback(false, ResponseData, ErrorMessage);
			}
			return;
		}

		// handle http status code
		int32 ResponseCode = HttpResponse.Get()->GetResponseCode();
		FString ResponseContent = HttpResponse.Get()->GetContentAsString();

		FString ResponseInfo = FString::Printf(
			TEXT("URL: %s\nResponseCode: %d\nResponseContent: %s"),
			*HttpRequest->GetURL(),
			ResponseCode,
			*ResponseContent
		);
		UE_LOG(LogTemp, Warning, TEXT("%s"), *ResponseInfo);

		if (ResponseCode < 200 || ResponseCode >= 300)
		{
			ErrorMessage = FString::Printf(
				TEXT("Manager Server Response Failed. %s"),
				*ResponseInfo
			);

			UE_LOG(LogTemp, Warning, TEXT("%s"), *ErrorMessage);

			if (Callback)
			{
				Callback(false, ResponseData, ErrorMessage);
			}
			return;
		}

		// handle coantent-string
		ResponseContent.TrimStartAndEndInline();
		if (ResponseContent.IsEmpty())
		{
			ResponseContent = TEXT("{}");
		}

		bool bParseResult = ParseJsonStringToStruct(ResponseContent, ResponseData, ErrorMessage);
			
		if (Callback)
		{
			Callback(bParseResult, ResponseData, ErrorMessage);
		}		
	});

	Request->ProcessRequest();
}

template<typename T>
bool FHiveManagerProtocol::ParseJsonStringToStruct(const FString& JsonString, T& OutStruct, FString& OutErrorMessage)
{
	if (JsonString.IsEmpty())
	{
		OutErrorMessage = TEXT("FHiveManagerProtocol::ParseJsonStringToStruct() Empty JSON string");
		return false;
	}

	bool bParseSuccess = false;

	if constexpr (TIsTArray<T>::Value)
	{
		bParseSuccess = FJsonObjectConverter::JsonArrayStringToUStruct(JsonString, &OutStruct);	
	}
	else
	{
		bParseSuccess = FJsonObjectConverter::JsonObjectStringToUStruct(JsonString, &OutStruct);	
	}
	
	if (!bParseSuccess)
	{
		OutErrorMessage = FString::Printf(
			TEXT("FHiveManagerProtocol::ParseJsonStringToStruct() Failed to parse JSON object. Content: %s"),
			*JsonString.Left(100)
		);
	}
	return bParseSuccess;
}

template<typename T>
void FHiveManagerProtocol::RequestGet(
	const FString& Url,
	FResponseCallback<T> Callback
) {
	TSharedRef<IHttpRequest, ESPMode::ThreadSafe> Request = FHttpModule::Get().CreateRequest();
	Request->SetURL(Url);
	Request->SetVerb(TEXT("GET"));

	SendRequest<T>(Request, Callback);
}

template<typename T>
void FHiveManagerProtocol::RequestPost(
	const FString& Url,
	TSharedPtr<FJsonObject> JsonBody,
	FResponseCallback<T> Callback
) {
	TSharedRef<IHttpRequest, ESPMode::ThreadSafe> Request = FHttpModule::Get().CreateRequest();
	Request->SetURL(Url);
	Request->SetVerb(TEXT("POST"));

	FString ContentString = TEXT("{}");
	if (JsonBody.IsValid()) {
		FString OutputString;
		TSharedRef< TJsonWriter<TCHAR, TCondensedJsonPrintPolicy< TCHAR >> > Writer = TJsonWriterFactory<TCHAR, TCondensedJsonPrintPolicy< TCHAR >>::Create(&OutputString);
		
		if (FJsonSerializer::Serialize(JsonBody.ToSharedRef(), Writer)) {
			ContentString = OutputString;
		}
	}
	Request->SetContentAsString(ContentString);

	SendRequest<T>(Request, Callback);
}